/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.action;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.SynchronousQueue;

import no.esito.log.Logger;

/**
 * An implementation of the ActionQueue. 
 */
public final class BlockingGuiActionQueue implements ActionQueue {

    /** The logger */
    private static final Logger log = Logger.getLogger(BlockingGuiActionQueue.class);
    
    /** Task queue, used to synchronize handing of task between threads. */
    private SynchronousQueue<TaskWrap<?>> taskQueue = 
        new SynchronousQueue<TaskWrap<?>>();
    
    @Override
    public final <V> V perform(Callable<V> task) throws InvocationTargetException {
        log.debug("Adding task to queue " + task);
        TaskWrap<V> taskWrap = new TaskWrap<V>(task);
        boolean interrupted = false;

        try {
            while (true) {
                try {
                    taskQueue.put(taskWrap);
                    break;
                } catch (InterruptedException e1) {
                    interrupted = true;
                }
            }
            while (true) {
                try {
                    return taskWrap.getResultValue();
                } catch (InterruptedException e) {
                    interrupted = true; 
                }
            }
        } finally {
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }

    @Override
    public final void perform(final Runnable task) throws InvocationTargetException {
        Callable<Void> callable = new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                task.run();
                return null;
            }

            /* (non-Javadoc)
             * @see java.lang.Object#toString()
             */
            @Override
            public String toString() {
                return task.toString();
            }
            
        };
        perform(callable);
    }

    @Override
    public final void ready() {
        log.debug("Ready to perform tasks.");
        boolean interrupted = false;
        try {
            while (true) {
                TaskWrap<?> taskWrap = null;
                try {
                    taskWrap = taskQueue.take();
                } catch (InterruptedException e) {
                    interrupted = true;
                }
                if (POISON == taskWrap) {
                    log.trace("Got poison pill");
                    break;
                }
                if (taskWrap != null) {
                    log.debug("Performing gui task: " + taskWrap);
                    taskWrap.performTask();
                }
            }
        } finally {
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
        log.debug("Finished perfoming tasks.");
    }

    @Override
    public final void release() {
        boolean interrupted = false;
        log.trace("About to post poison pill.");
        try {
            while (true) {
                try {
                    taskQueue.put(POISON);
                    break;
                } catch (InterruptedException e) {
                    log.trace("Interrupted!");
                    interrupted = true;
                }
            }
        } finally {
            log.trace("Posted poison pill.");
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }

    /**
     * Wraps a callable. In order to perform on one thread and return the
     * result on an other, the invoking thread must store the callable result
     * in a resultValue that can be returned by the other thread. 
     *
     * @param <V> the callable result value type.
     */
    private static class TaskWrap<V> {
        private Callable<V> task;
        private V resultValue;
        private CountDownLatch gate = new CountDownLatch(1);
        private volatile Exception caughtException = null;
        
        TaskWrap(Callable<V> task) {
            this.task = task;
        }
        
        final void performTask() {
            log.debug("Performing task " + task);
            try {
                resultValue = task.call();
            } catch (Exception e) {
                log.warn("Caught exception during action execution", e);
                caughtException = e;
            } finally {
                gate.countDown();
            }
        }

        final V getResultValue() throws InterruptedException, InvocationTargetException {
            gate.await();
            if (caughtException != null) {
                throw new InvocationTargetException(caughtException);
            }
            return resultValue;
        }

        @Override
        public String toString() {
            return task.toString();
        }
    }
    
    private final TaskWrap<Void> POISON = new TaskWrap<Void>(new Callable<Void>() {
        @Override
        public Void call() throws Exception {
            return null;
        }
    } );
    
}
