/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.action;

import java.util.concurrent.CountDownLatch;

import no.esito.log.Logger;
import no.g9.client.core.message.BlockingMessageCallback;
import no.g9.client.core.message.InteractionThreadPolicy;
import no.g9.message.Message;

/**
 * A message callback implementation that is suitable for a two-threaded model
 * such as the Icefaces target and possibly other.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public class UAMessageCallback implements BlockingMessageCallback {

    private volatile Message message = null;

    private final static Logger log = Logger.getLogger(UAMessageCallback.class);

    private CountDownLatch latch;

    private ActionQueue guiActionQueue;

    private synchronized CountDownLatch getLatch() {
        if (latch == null) {
            latch = new CountDownLatch(1);
        }
        return latch;
    }

    private synchronized void removeLatch() {
        latch = null;
    }

    /**
     * Returns the reply set by the UA thread. This method blocks until a reply
     * exists.
     *
     * @return the message with reply.
     */
    @Override
    public Message getMessage() {

        if (log.isTraceEnabled()) {
            log.trace("Getting message.");
        }

        message = null;

        guiActionQueue.release();

        boolean interrupted = false;
        try {
            while (message == null) {
                try {
                    if (log.isTraceEnabled()) {
                        log.trace("No message received yet - going to sleep.");
                    }
                    getLatch().await();
                    if (log.isTraceEnabled()) {
                        log.trace("Woke up");
                    }
                } catch (InterruptedException e) {
                    log.debug("Caught interrupted exception:", e);
                    interrupted = true;
                    // fall through and retry
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("Returning message " + message);
            }
            return message;
        } finally {
            removeLatch();
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }


    }

    /*
     * Updates the thread shared message instance and puts the invoking thread
     * to sleep on the uaBlocker instance.
     * @see no.g9.message.MessageCallback#reply(no.g9.message.Message)
     */
    @Override
    public void reply(Message msg) {

        if (log.isTraceEnabled()) {
            log.trace("Received message reply " + msg);
        }

        synchronized (this) {
            this.message = msg;
        }

        getLatch().countDown();

        guiActionQueue.ready();



    }

    /**
     * Sets the quiActionQuque that will be used to coordinate the UA thread.
     *
     * @param guiActionQueue the coordinating action queue.
     */
    public final void setGuiActionQueue(ActionQueue guiActionQueue) {
        this.guiActionQueue = guiActionQueue;
    }

    @Override
    public InteractionThreadPolicy getInteractionThreadPolicy() {
        return InteractionThreadPolicy.ASYNC_WORKER_POLICY;
    }


}