/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import no.esito.jvine.rpc.ActualParameter;
import no.esito.jvine.rpc.ActualParameter.ParameterType;
import no.esito.jvine.view.AbstractApplicationView;
import no.esito.log.Logger;
import no.g9.client.core.action.ActionTask;
import no.g9.client.core.action.CheckType;
import no.g9.client.core.action.Parameter;
import no.g9.client.core.action.ParameterBinding;
import no.g9.client.core.action.RemoteServiceTarget;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.converter.FrameworkConversionException;
import no.g9.client.core.validator.ValidateContext;
import no.g9.client.core.validator.ValidationException;
import no.g9.client.core.validator.ValidationResult;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.Resource;
import no.g9.client.core.view.ViewModel;
import no.g9.client.core.view.table.TableModel;
import no.g9.exception.G9BaseException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.MessageReply;
import no.g9.message.MessageReplyType;
import no.g9.os.AttributeConstant;
import no.g9.os.Key;
import no.g9.os.KeyTool;
import no.g9.os.OSRole;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.ObjectSelection;
import no.g9.support.action.ActionTarget;

/**
 * Utility class. Used to create various action tasks.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
class ActionTaskFactory {
    private static final class CheckTask extends ActionTask<Boolean> {
        private final DialogController dialogController;
        private final ActionType checkType;
        final OSNode<?>[] targetNode;
        final RoleConstant targetRoleConstant;
        final DialogController targetDialog;

        private CheckTask(DialogController dialogController, Enum<?> target,
                ActionType checkType) {
            this.dialogController = dialogController;
            this.checkType = checkType;
            targetNode = new OSNode<?>[1];
            if (target instanceof RoleConstant) {
                targetNode[0]= dialogController.getOSRole((RoleConstant) target);
                targetRoleConstant= (RoleConstant) target;
            } else if (target instanceof DialogObjectConstant) {
                DialogObjectConstant doc= (DialogObjectConstant) target;
                targetNode[0]= dialogController.getOSRole(doc.getRole());
                targetRoleConstant= doc.getRole();
            } else {
                targetNode[0]= null;
                targetRoleConstant= null;
            }
            targetDialog = null;
        }

        private CheckTask(DialogController dialogController,
                DialogConstant target, ActionType checkType) {
            this.dialogController = dialogController;
            this.checkType = checkType;
            targetNode = null;
            targetRoleConstant = null;
            targetDialog = dialogController.getApplicationController()
                    .getDialogController(target);

        }

        private CheckTask(DialogController dialogController,
                RoleConstant target, ActionType checkType) {
            this.dialogController = dialogController;
            this.checkType = checkType;
            targetNode = JVineController.getInstance(dialogController)
                    .getRootNodes();
            targetRoleConstant = target;
            targetDialog = null;
        }

        private CheckTask(DialogController dialogController,
                RemoteServiceTarget target) {
            this.dialogController = dialogController;
            this.checkType = ActionType.INVOKE;
            targetNode = RoleLocator.getNodesToCheck(dialogController, target);
            targetRoleConstant = null;
            targetDialog = dialogController;
        }

        @Override
        @SuppressWarnings("deprecation")
        public Boolean call() throws Exception {
            if (checkType == null) {
                return Boolean.TRUE;
            }
            String msgID = null;
            String msgDesc = null;
            boolean checkOK = true;
            CheckResult checkResult = null;
            switch (checkType) {

            case CHECK_FIND:
                for (OSNode<?> target : targetNode) {
                    CheckResult tmp = target.checkFind();
                    if (checkResult == null) {
                        checkResult = tmp;
                    } else {
                        checkResult
                                .addConversionError(tmp.getConversionError());
                        if (!tmp.getCheckResult()) {
                            checkResult.setCheckResult(tmp.getCheckResult());
                        }
                    }

                    checkOK = checkOK && checkResult.getCheckResult();
                }
                if (checkResult != null
                        && !checkResult.getConversionError().isEmpty()) {
                    throw new FrameworkConversionException(
                            checkResult.getConversionError());
                }
                msgID = CRuntimeMsg.CC_CHECK_FIND_MSG;
                msgDesc = "Dialog changed, continue with find?";
                break;
            case CHECK_DELETE:
                for (OSNode<?> target : targetNode) {
                    CheckResult cr = target.checkChanged(CheckType.DELETE,
                            targetRoleConstant);
                    if (checkResult == null) {
                        checkResult = cr;
                    }
                    checkOK = checkOK && cr.getCheckResult();
                    checkResult.addConversionError(cr.getConversionError());
                    checkResult.setCheckResult(checkOK);
                }
                msgID = CRuntimeMsg.CC_CHECK_DELETE_MSG;
                msgDesc = "Dialog changed, continue with delete?";

                if (checkResult != null
                        && !checkResult.getConversionError().isEmpty()) {
                    throw new FrameworkConversionException(
                            checkResult.getConversionError());
                }
                break;
            case INSERT:
                // fall through
            case UPDATE:
                // fall through
            case SAVE:
                boolean retVal = true;
                for (OSNode<?> target : targetNode) {
                    CheckResult checkSaveResult = target.checkSave(checkType);
                    Map<ValidationResult, ValidateContext> validationResult = checkSaveResult
                            .getValidationResult();

                    if (!validationResult.isEmpty()) {
                        throw new ValidationException(validationResult);
                    }
                    Map<DialogObjectConstant, Collection<?>> conversionError = checkSaveResult
                            .getConversionError();
                    if (!conversionError.isEmpty()) {
                        throw new FrameworkConversionException(conversionError);
                    }
                    retVal = retVal && checkSaveResult.getCheckResult();
                }
                return Boolean.valueOf(retVal);
            case CHECK_CLOSE:
                checkOK = JVineController.getInstance(targetDialog)
                        .checkClose();
                msgID = CRuntimeMsg.CC_CHECK_CLOSE_MSG;
                DialogInstanceKey instance = new DialogInstanceKey(targetDialog.getDialogConstant(), targetDialog.getDialogInstanceNumber());
				msgDesc = dialogController.getApplicationController().getApplicationView().getDialogTitle(instance);
                break;
            case SELECT:
                for (OSNode<?> target : targetNode) {
                    checkOK = checkOK
                            && target.checkChanged(CheckType.CLEAR,
                                    targetRoleConstant).getCheckResult();
                }
                msgID = CRuntimeMsg.CC_CHECK_SELECTION_MSG;
                msgDesc = "Unsaved data will be lost! Select new row?";
                break;
            case UNSELECT:
                for (OSNode<?> target : targetNode) {
                    checkOK = checkOK
                            && target.checkChanged(CheckType.CLEAR,
                                    targetRoleConstant).getCheckResult();
                }
                msgID = CRuntimeMsg.CC_CHECK_UNSELECTION_MSG;
                msgDesc = "Unsaved data will be lost! Unselect row?";
                break;
            case CHECK_CLEAR:
                for (OSNode<?> target : targetNode) {
                    checkOK = checkOK
                            && target.checkChanged(CheckType.CLEAR,
                                    targetRoleConstant).getCheckResult();
                }
                msgID = CRuntimeMsg.CC_CHECK_CLEAR_MSG;
                msgDesc = "Dialog changed, continue with clear?";
                break;
            case INVOKE:
                for (OSNode<?> target : targetNode) {
                    CheckResult cr = target.checkChanged(CheckType.INVOKE,
                            target.getRoleConstant());
                    checkOK = cr.getCheckResult();
                    Map<DialogObjectConstant, Collection<?>> messages = cr
                            .getConversionError();
                    if (!messages.isEmpty()) {
                        throw new FrameworkConversionException(messages);
                    }
                }
                msgID = CRuntimeMsg.CC_CHECK_INVOKE_MSG;
                msgDesc = "Dialog changed, continue?";
                break;

            default:
                checkOK = true;
            }
            return Boolean.valueOf(checkOK || interact(msgID, msgDesc));

        }

        boolean interact(String msgID, String msgDesc) {
            MessageReply reply = dialogController.dispatchMessage(msgID,
                    msgDesc);
            return MessageReplyType.REPLY_OK.equals(reply);
        }
    }

    private final static Logger log = Logger.getLogger(ActionTaskFactory.class);

    /**
     * Creates a new grape action.
     *
     * @param controller the dialog controller of said action
     * @param action the action type
     * @return a {@link GrapeActionTask}
     */
    static ActionTask<Object> createGrapeAction(DialogController controller,
            ActionType action) {
        switch (action) {
        case FIND: // fall through
        case FINDALL: // fall through
        case INSERT: // fall through
        case DELETE: // fall through
        case SAVE: // fall through
        case UPDATE:
            return new GrapeActionTask(controller, action);
        default:
            String msg = "Excpected grape action, got " + action;
            throw new IllegalArgumentException(msg);
        }
    }

    /**
     * Create a new export action.
     *
     * @param target the target dialog of the pdf export
     * @param ctrl application controller
     * @return an {@link ExportAction}
     */
    static ActionTask<Resource> createExportAction(DialogConstant target,
            ApplicationController ctrl) {
        return new ExportAction(target, ctrl);

    }

    /**
     * Create a new spreadsheet action.
     *
     * @param target the target dialog of the spreadsheet export
     * @param ctrl application controller
     * @return an {@link SpreadsheetAction}
     */
    static ActionTask<Resource> createSpreadsheetAction(DialogConstant target, ApplicationController ctrl) {
    	return new SpreadsheetAction(target, ctrl);
    }

    /**
     * Creates a new clear task for the specified target.
     *
     * @param target the target to clear
     * @param dialogController the dialog controller
     * @return an action task set up to clear the specified target
     */
    static ActionTask<Void> createClearTask(final Enum<?> target,
            final DialogController dialogController) {

        ActionTask<Void> actionTask = new ActionTask<Void>() {

            @Override
            public Void call() throws Exception {
                JVineController jVineController = JVineController
                        .getInstance(dialogController);
                OSNode<?>[] nodeTargets;
                RoleConstant targetRoleConstant = (RoleConstant) target;
                if (targetRoleConstant.isObjectSelection()) {
                    nodeTargets = jVineController.getRootNodes();
                } else {
                    OSNode<?> node = dialogController
                            .getOSRole((RoleConstant) target);
                    nodeTargets = new OSNode<?>[] { node };
                }
                for (OSNode<?> targetNode : nodeTargets) {
                    jVineController.reportToView(targetNode.clear(true),
                            jVineController.getOneRelatedChildren(targetNode
                                    .getRoleConstant()), null);
                }
                return null;
            }

        };
        return actionTask;
    }

    /**
     * Creates a new activate task.
     *
     * @param target the method to activate
     * @return a dummy action task for the activate action
     */
    static ActionTask<ActionTarget> createActivateTask(final ActionTarget target) {
    	
    	ActionTask<ActionTarget> actionTask = new ActionTask<ActionTarget>() {
    		
    		@Override
    		public ActionTarget call() throws Exception {
    			return target;
    		}
    		
    	};
    	return actionTask;
    }
    
    /**
     * Creates an action task capable of displaying data in a dialog.
     *
     * @param target the role to display
     * @param actionType the action type for this display
     * @param dialogController the dialog controller of the dialog to display to
     * @return the task capable of performing said task
     */
    static ActionTask<Void> createDisplayDialogTask(Object target, ActionType actionType, DialogController dialogController) {
        if (target instanceof Enum<?>) {
            return createDisplayDialogTask((Enum<?>) target, actionType, dialogController);
        }
        if (target instanceof RemoteServiceTarget) {
            return createDisplayRemoteServiceResult((RemoteServiceTarget) target, dialogController);
        }

        // Unknown target - log and throw.
        String msg = "Unknown target " + target;
        throw new RuntimeException(msg);
    }

    private static ActionTask<Void> createDisplayRemoteServiceResult(
            final RemoteServiceTarget target,
            final DialogController dialogController) {
        return new ActionTask<Void>() {

            @Override
            public Void call() throws Exception {
                if (log.isDebugEnabled()) {
                    log.debug(this + " displaying " + target);
                }
                ParameterBinding<?> returnParameter = target
                        .getReturnParameter();
                Object value = getTaskObject();
                Parameter<?> parameter = returnParameter.getActualParameter();
                if (parameter instanceof ActualParameter<?>) {
                    ActualParameter<?> actualParameter = (ActualParameter<?>) parameter;
                    ParameterType parameterType = actualParameter
                            .getParameterType();
                    switch (parameterType) {
                    case UNBOUND:
                        break;
                    case ATTRIBUTE:
                        AttributeConstant attribute = actualParameter
                                .getAttribute();
                        dialogController.setFieldValue(attribute, value);
                        break;
                    case ROLE:
                        RoleConstant role = actualParameter.getRole();
                        OSNode<?> node = dialogController.getOSRole(role);
                        Collection<OSNode<?>> changedNodes = null;
                        if (value instanceof Collection<?>) {
                            changedNodes = node.setCurrentInstances((Collection<?>)value);
                        } else {
                            changedNodes = node.setCurrentInstance(value);
                        }
                        Collection<OSNode<?>> include = new HashSet<OSNode<?>>();
                        include.add(node);
                        JVineController.getInstance(dialogController).reportToView(
                                changedNodes, include, null);
                        break;
                    default:
                        String msg = "Can't display actual parameter "
                                + actualParameter;
                        throw new RuntimeException(msg);
                    }
                }

                return null;
            }

        };
    }

    /**
     * Gets the action task that recursively displays on a node
     *
     * @param target the target node
     * @param actionType the action type for this display
     * @param dialogController the dialog controller
     * @return an action task capable of displaying from the target node
     */
    private static ActionTask<Void> createDisplayDialogTask(final Enum<?> target, final ActionType actionType, final DialogController dialogController) {
        final OSNode<?> targetNode = dialogController.getOSRole((RoleConstant) target);
        return new ActionTask<Void>() {

            @Override
            public Void call() throws Exception {
                if (log.isDebugEnabled()) {
                    log.debug(this + " displaying " + target);
                }
                Collection<?> result = (Collection<?>) getTaskObject();
                newDisplay(targetNode, actionType, result);
                return null;
            }

            private void newDisplay(OSNode<?> actionTarget, ActionType actionType, Collection<?> result) {
                if (actionType == ActionType.FINDALL) {
                    // target must be root...
                    Collection<OSNode<?>> changedNodes = actionTarget.setCurrentInstances(result);
                    Collection<OSNode<?>> include = new HashSet<OSNode<?>>();
                    include.add(targetNode);
                    JVineController.getInstance(dialogController).reportToView(changedNodes, include, null);
                    sortAndFilter(dialogController, changedNodes, include);
                } else {
                    Object newCurrent = wash(result);
                    if (newCurrent != null) {
                        Collection<OSNode<?>> changedNodes = null;
                        if (isSaveParentAction(actionType, targetNode)) {
                            changedNodes = actionTarget.setCurrentInstance(null);
                            getRootNode(targetNode).setCurrentInstance(newCurrent);
                            changedNodes.addAll(getRootPathNodes(targetNode));
                        }
                        else {
                            changedNodes = actionTarget.setCurrentInstance(newCurrent);
                        }
                        Collection<OSNode<?>> include = new HashSet<OSNode<?>>();
                        include.add(targetNode);
                        JVineController.getInstance(dialogController).reportToView(changedNodes, include, null);
                        sortAndFilter(dialogController, changedNodes, include);
                    }
                }
            }

            private <R> R wash(Collection<R> collection) {
                if (collection == null) {
                    return null;
                }
                if (collection.isEmpty()) {
                    return null;
                }
                return collection.iterator().next();
            }


            /**
             * @param dialogController
             * @param changedNodes
             * @param includeNodes
             */
            private void sortAndFilter(final DialogController dialogController,
                    Collection<OSNode<?>> changedNodes,
                    Collection<OSNode<?>> includeNodes) {
                Collection<OSNode<?>> washedList = CtrlObservable.washList(
                        changedNodes, includeNodes, null);
                ViewModel viewModel = dialogController.getDialogView()
                        .getViewModel();
                for (OSNode<?> node : washedList) {
                    TableModel<ListRow> tableModel = viewModel
                            .getTableModel(node.getRoleConstant());
                    if (tableModel != null) {
                        tableModel.filterTableView();
                        tableModel.sortTableView();
                    }
                }
            }

            private Collection<OSNode<?>> getRootPathNodes(OSNode<?> target) {
                Collection<OSNode<?>> rootPathNodes = new HashSet<OSNode<?>>();
                OSNode<?> rootRole = target;
                JVineController jctrl = JVineController.getInstance(dialogController);
                while (rootRole.getParent() != null) {
                    OSRole<?> parent = rootRole.getParent();
                    rootRole = jctrl.getOSNode(parent.getRoleConstant());
                    rootPathNodes.add(rootRole);
                }
                return rootPathNodes;
            }

            private OSNode<?> getRootNode(OSNode<?> target) {
                OSNode<?> rootRole = target;
                JVineController jctrl = JVineController.getInstance(dialogController);
                while (rootRole.getParent() != null) {
                    OSRole<?> parent = rootRole.getParent();
                    rootRole = jctrl.getOSNode(parent.getRoleConstant());
                }
                return rootRole;
            }

        };
    }

    /**
     * Gets the obtain dialog task for the specified controller
     *
     * @param target the target to obtain
     * @param dialogController the dialog controller to obtain from
     * @param actionType the action type for this obtain
     * @return the action task capable of obtaining the specified target
     */
    static ActionTask<ObjectSelection> createObtainDialogTask(Object target, DialogController dialogController, ActionType actionType) {
        if (target instanceof Enum<?>) {
            return createObtainDialogTask((Enum<?>) target, dialogController, actionType);
        }
        if (target instanceof RemoteServiceTarget) {
            return createObtainDialogTask((RemoteServiceTarget) target,
                    dialogController);
        }

        String msg = "Unknown target " + target;
        throw new RuntimeException(msg);
    }

    private static OSNode<?>[] getObtainTarget(
            DialogController dialogController, Enum<?> target) {
        if (target instanceof RoleConstant) {
            return new OSNode[] { dialogController
                    .getOSRole((RoleConstant) target) };
        }

        if (target instanceof DialogConstant) {
            // This is a print action....
            return JVineController.getInstance(dialogController).getRootNodes();
        }

        throw new G9BaseException("Can't obtain target: " + target);
    }

    /**
     * Gets the obtain dialog task for the specified controller
     *
     * @param target the target to obtain
     * @param dialogController the dialog controller to obtain from
     * @param actionType the action type for this obtain
     * @return the action task capable of obtaining the specified target
     */
    private static ActionTask<ObjectSelection> createObtainDialogTask(final Enum<?> target, final DialogController dialogController, final ActionType actionType) {

        final OSNode<?> node[] = getObtainTarget(dialogController, target);
    	final boolean clearCurrent = !isSaveParentAction(actionType, node[0]) && ActionType.DELETE == actionType;
        return new ActionTask<ObjectSelection>() {

            @Override
            public ObjectSelection call() throws Exception {
                Object actionTarget = node[0].obtain(clearCurrent);
                OSNode<?> root[] = new OSNode[node.length];
                for (int i = 0; i < node.length; i++) {
                    root[i] = node[i];
                }
                JVineController jctrl = JVineController.getInstance(dialogController);
                for (int i = 0; i < root.length; i++) {
                    OSNode<?> rootRole = root[i];
                    while (rootRole.getParent() != null) {
                        OSRole<?> parent = rootRole.getParent();
                        RoleConstant roleConstant = parent.getRoleConstant();
                        rootRole = jctrl.getOSNode(roleConstant);
                    }
                    root[i] = rootRole;

                }

                String targetName = node[0].getRoleConstant().toString();
                String applicationName = dialogController.getApplicationController().getApplicationName();
                String objectSelectionName = dialogController.getObjectSelectionName();
                ObjectSelection os = new ObjectSelection(actionTarget, targetName, targetName, applicationName, objectSelectionName);
                Key key = KeyTool.getCompleteUniqueKey(actionTarget, node[0]);
                if (key != null) {
                    os.setTargetMainKey(key.getKeyName());
                }

                // Obtain all roots for print targets
                if (target instanceof DialogConstant) {
                    for (int i = 0; i < root.length; i++) {
                        List<Object> rootInstances = getRootInstances(root[i]);
                        os.putRootObjects(root[i].toString(), rootInstances);
                    }
                }
                else if (isSaveParentAction(actionType, node[0])) {
                	List<Object> rootInstances = getRootInstances(root[0]);
                	if (actionTarget == null) {
                	    rootInstances.clear();
                	    rootInstances.add(root[0].obtain(false));
                	}
                	String rootName = root[0].getRoleConstant().toString();
                	os.removeAllRootObjects();
					os.putRootObjects(rootName, rootInstances);
                	os.setCurrentRootName(rootName);
                	// Set the action map for the target object to avoid stripping before Jgrape call
                	Map<Object, ActionType> targetActions = new HashMap<>();
                	if (actionTarget != null) {
                	    targetActions.put(actionTarget, actionType);
                	}
                	os.setActions(targetActions);
                }
                return os;
            }

            private List<Object> getRootInstances(OSNode<?> node) {
                List<Object> list = new ArrayList<Object>();
                Collection<?> allInstances = node.getAllInstances();
                if (allInstances.size() > 0) {
                    list.addAll(allInstances);
                } else {
                    list.add(node.obtain(false));
                }
                return list;
            }

        };
    }

    /**
     * Gets the obtain dialog task for the specified controller
     *
     * @param target the target to obtain
     * @param dialogController the dialog controller to obtain from
     * @return the action task capable of obtaining the specified target
     */
    private static ActionTask<ObjectSelection> createObtainDialogTask(
            final RemoteServiceTarget target,
            final DialogController dialogController) {
        return new ActionTask<ObjectSelection>() {

            @Override
            public ObjectSelection call() throws Exception {
                ObjectSelection os = new ObjectSelection();
                String applicationName = dialogController
                        .getApplicationController().getApplicationName();
                os.setApplicationName(applicationName);
                String targetName = target.getService().toString();
                os.setTargetRoleName(targetName);
                os.putRootObject(targetName, target);

                List<ParameterBinding<?>> parameterBindings = target
                        .getParameterBindings();
                for (ParameterBinding<?> parameterBinding : parameterBindings) {
                    parameterBinding.obtainParameterValue();
                }
                return os;
            }

        };
    }

    /**
     * Creates a new dialog action task capable of performing the specified
     * action.
     *
     * @param action the action to perform
     * @param target the target of said action
     * @param controller the dialog controller invoking the action
     * @return the dialog action task
     */
    static ActionTask<Void> createDialogActionTask(final ActionType action,
            final DialogConstant target, final DialogController controller) {
        ActionTask<Void> actionTask = new ActionTask<Void>() {

            @Override
            public Void call() throws Exception {
                Object taskObject = getTaskObject();
                boolean tmp = false;
                try {
                    tmp = (Boolean) taskObject;
                } catch (NullPointerException npe) {
                    tmp = false;
                } catch (ClassCastException cce) {
                    tmp = false;
                }
                controller.getApplicationController().performAction(action, target, tmp);
                if (canTriggerShownHiddenEvent()) {
                	getApplicationView().triggerShownHiddenEvent(target, action);
                }
                return null;
            }

            private boolean canTriggerShownHiddenEvent() {
            	if (controller.getDialogConstant() == target) {
            		return false;
            	}
            	return true;
            }

            private AbstractApplicationView getApplicationView() {
            	return (AbstractApplicationView) controller.getApplicationController().getApplicationView();
            }

        };

        return actionTask;
    }

    /**
     * Creates a new dialog action task capable of performing the specified
     * action. The action source is external.
     *
     * @param action the action to perform
     * @param target the target of said action
     * @param controller the dialog controller invoking the action
     * @return the dialog action task
     */
    static ActionTask<Void> createDialogActionTaskFromExternal(final ActionType action,
            final DialogConstant target, final DialogController controller) {
        ActionTask<Void> actionTask = new ActionTask<Void>() {

            @Override
            public Void call() throws Exception {
                Object taskObject = getTaskObject();
                boolean tmp = false;
                try {
                    tmp = (Boolean) taskObject;
                } catch (NullPointerException npe) {
                    tmp = false;
                } catch (ClassCastException cce) {
                    tmp = false;
                }
                controller.getApplicationController().performActionFromExternal(action, target, tmp);
                return null;
            }
        };

        return actionTask;
    }

    /**
     * Creates a new gui action task capable of performing the specified action.
     *
     * @param action the action to perform
     * @param target the target of the action
     * @param dialogController the dialog controller invoking the action
     * @return the gui task
     */
    static ActionTask<Void> createGuiActionTask(final ActionType action,
            final ActionTarget target, final DialogController dialogController) {
        ActionTask<Void> actionTask = new ActionTask<Void>() {

            @Override
            public Void call() throws Exception {
                dialogController.<DialogView> getDialogView().performAction(
                        action, (DialogObjectConstant) target);
                return null;
            }

        };
        return actionTask;
    }

    /**
     * Gets the action task that will perform a dialog check.
     *
     * @param checkType the check type
     * @param target the target to check
     * @param dialogController the dialog controller
     * @return the task that will perform the check
     */
    static ActionTask<Boolean> createCheckDialogTask(
            final ActionType checkType, final Object target,
            final DialogController dialogController) {

        if (checkType == ActionType.OPEN) {
            return new ActionTask<Boolean>() {

                @Override
                public Boolean call() throws Exception {
                    boolean shouldTest = FLAG;
                    if (!shouldTest) {
                        log.debug("Possible reuse of " + target
                                + " instance - no test for avilability needed.");
                        return true;
                    }

                    log.debug("Testing that application allows opening of "
                            + target);

                    boolean hasAvailableDialogInstance = true;
                    hasAvailableDialogInstance = dialogController
                            .getApplicationController()
                            .hasAvailableDialogInstance((DialogConstant) target);
                    log.debug("Application controller has available dialog instance for "
                            + target + " = " + hasAvailableDialogInstance);
                    return hasAvailableDialogInstance;
                }

            };
        }
        if (target instanceof DialogConstant) {
            return new CheckTask(dialogController, (DialogConstant) target,
                    checkType);
        }

        if (target instanceof RoleConstant) {
            RoleConstant targetRole = (RoleConstant) target;
            if (targetRole.isObjectSelection()) {
                return new CheckTask(dialogController, targetRole, checkType);
            }

        }

        if (target instanceof RemoteServiceTarget) {
            return new CheckTask(dialogController, (RemoteServiceTarget) target);
        }

        return new CheckTask(dialogController, (Enum<?>) target, checkType);
    }

    private static boolean isSaveParentAction(ActionType actionType, OSNode<?> targetNode) {
        if (targetNode.isRoot()) {
            return false;
        }
        switch (actionType) {
            case SAVE:
            case INSERT:
            case UPDATE:
            case DELETE:
                return true;
            default:
                return false;
        }
    }

}
