/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;

import no.esito.jvine.controller.OSNode;
import no.esito.log.Logger;
import no.g9.client.core.view.DialogView;

/**
 * Wraps observers around dialog views and offers convenient methods for
 * notifying the observers.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
class CtrlObservable extends Observable {

    private final static Logger log = Logger.getLogger(CtrlObservable.class);

    private Map<DialogView, Observer> viewToObserver = new HashMap<DialogView, Observer>();

    @Override
    protected synchronized void setChanged() {
        super.setChanged();
    }

    /**
     * Registers a dialog view as an observer.
     * 
     * @param dialogView the dialog view wanting to be notified when this
     *            observable changes.
     */
    void registerView(final DialogView dialogView) {
        Observer observer = new Observer() {

            @Override
            @SuppressWarnings("unchecked")
            public void update(Observable o, Object arg) {
                dialogView.update((Collection<OSNode<?>>) arg);
            }
        };

        viewToObserver.put(dialogView, observer);

        addObserver(observer);
    }

    /**
     * Removes a previously registered view from this observable.
     * 
     * @param dialogView the dialog view to remove.
     */
    void removeView(DialogView dialogView) {
        deleteObserver(viewToObserver.get(dialogView));
    }

    /**
     * Report changed nodes to the view. The <code>changedNodes</code>
     * collection can consist of arbitrary nodes, but only roots and many-related will be reported 
     * to the view. Nodes in the <code>includeNodes</code> collection are always reported.
     * Nodes in the <code>excludeNodes</code> collection are always excluded.
     * 
     * @param changedNodes arbitrary list of changed nodes.
     * @param includeNodes nodes that is always reported.
     * @param excludeNode nodes that are always excluded.
     */
    void reportToView(Collection<OSNode<?>> changedNodes,
            Collection<OSNode<?>> includeNodes, OSNode<?> excludeNode) {
        if (log.isTraceEnabled()) {
            log.trace("Reporting changed nodes to view.");
            log.trace("Complete list before washing is: " + changedNodes);
            log.trace("Included nodes are: " + includeNodes);
            log.trace("Exclude node is: " + excludeNode);
        }

        Collection<OSNode<?>> changedManyNodes = washList(changedNodes,
                includeNodes, excludeNode);

        setChanged();
        if (log.isDebugEnabled()) {
            log.debug("Notifying observers about changed nodes: "
                    + changedManyNodes);
        }
        notifyObservers(changedManyNodes);
    }
    
    /**
     * Washes the list. 
     * @param changedNodes changed nodes
     * @param includeNodes nodes to include
     * @param excludeNode nodes to exclude
     * @return collection of changed nodes
     */
    static Collection<OSNode<?>> washList(Collection<OSNode<?>> changedNodes,
            Collection<OSNode<?>> includeNodes, OSNode<?> excludeNode) {

        if (log.isTraceEnabled()) {
            log.trace("Washing list of changed nodes reported to view.");
        }

        Collection<OSNode<?>> changedManyNodes = new HashSet<OSNode<?>>();

        for (OSNode<?> node : changedNodes) {
            switch (node.getRelationCardinality()) {
                case MANY: // fall through
                case ROOT:
                    changedManyNodes.add(node);
                    break;
                default:
                    break;
            }
        }

        if (includeNodes != null) {
            changedManyNodes.addAll(includeNodes);
        }

        if (excludeNode != null && changedManyNodes.contains(excludeNode)) {
            changedManyNodes.remove(excludeNode);
        }

        return changedManyNodes;
    }

}
