/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.io.Serializable;

import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogInstance;

import org.springframework.util.Assert;

/**
 * Identifies a dialog instance based on the dialog constant and a dialog
 * instance number.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public class DialogInstanceKey implements DialogInstance, Serializable {

    private final DialogConstant dialogConstant;
    private final int dialogInstanceNumber;
    
    /**
     * Constructs a new dialog instance key that wraps the specified dialog
     * constant and instancenumber.
     * 
     * @param dialogConstant
     *            the dialog constant denoting the dialog controller.
     * @param dialogInstanceNumber
     *            the dialog instance number
     */
    public DialogInstanceKey(DialogConstant dialogConstant, int dialogInstanceNumber) {
        int maxNumberOfInstances = dialogConstant.getMaximumNumberOfInstances() <= 0 ? 1 : dialogConstant
                .getMaximumNumberOfInstances();
        Assert.notNull(dialogConstant, "dialogConstant must not be null");
        Assert.isTrue(dialogInstanceNumber >= 1 && dialogInstanceNumber <= maxNumberOfInstances,
                "dialogInstanceNumber must be >= 1 and <= " + maxNumberOfInstances + ", was: " + dialogInstanceNumber);
        this.dialogConstant = dialogConstant;
        this.dialogInstanceNumber = dialogInstanceNumber <= 0 ? 1 : dialogInstanceNumber;

    }

    /**
     * Gets the dialog constant.
     * 
     * @return the dialog constant
     */
    @Override
    public DialogConstant getDialogConstant() {
        return dialogConstant;
    }

    @Override
    public int getDialogInstanceNumber() {
        return dialogInstanceNumber;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((dialogConstant == null) ? 0 : dialogConstant.hashCode());
        result = prime * result + dialogInstanceNumber;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (!(obj instanceof DialogInstanceKey))
            return false;
        DialogInstanceKey other = (DialogInstanceKey) obj;
        if (dialogConstant == null) {
            if (other.dialogConstant != null)
                return false;
        } else if (!dialogConstant.equals(other.dialogConstant))
            return false;
        if (dialogInstanceNumber != other.dialogInstanceNumber)
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "DialogInstanceKey [dialogConstant=" + dialogConstant + ", dialogInstanceNumber=" + dialogInstanceNumber
                + "]";
    }

}
