/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.util.Collection;
import java.util.List;

import no.esito.jvine.action.HookProviderImpl;
import no.esito.log.Logger;
import no.esito.util.BeanID;
import no.esito.util.ServiceLoader;
import no.g9.client.core.action.DialogInteraction;
import no.g9.client.core.controller.DialogCallBackValue;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogSetupValue;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;

/**
 * The dialog interaction broker is used to control the invocation of dialog
 * interaction hooks.
 */
@BeanID("dialogInteractionBroker")
public class DialogInteractionBroker {

    /** The logger */
    private static final Logger log = Logger
            .getLogger(DialogInteractionBroker.class);

    private DialogInteractionBroker() {
        // Use factory method...
    }

    /**
     * Runs the provided DialogSetupValue through all hooks.
     *
     * @param provided
     *            initial dialog setup value
     * @param caller
     *            the dialog that performs the <em>open</em> action
     * @param callee
     *            the dialog that will be <em>opened</em>
     * @return the DialogSetupValue
     */
    public DialogSetupValue<?> doSetup(DialogSetupValue<?> provided,
            DialogController caller, DialogConstant callee) {

        if (log.isDebugEnabled()) {
            log.debug("Interaction hooks for interaction setup between caller:" + caller
                    + ", and callee: " + callee
                    + ". \n\tProvided setup value is: " + provided);
        }

        Collection<DialogInteraction> interactionHook = getInteractionHook(
                caller, callee);
        for (DialogInteraction dialogInteraction : interactionHook) {
            provided = dialogInteraction
                    .getSetupValue(provided, caller, callee);
        }

        if (log.isTraceEnabled()) {
            log.trace("After running provided setup value through hooks, the value is: "
                    + provided);
        }

        return provided;
    }

    /**
     * Runs the provided DialogCallBackValue through all hooks.
     *
     * @param provided
     *            initial dialog setup value
     * @param caller
     *            the dialog that performed the <em>open</em> action
     * @param callee
     *            the dialog that was <em>opened</em>
     * @return the DialogSetupValue
     */
    public DialogCallBackValue doCallback(DialogCallBackValue provided,
            DialogController caller, DialogConstant callee) {

        if (log.isDebugEnabled()) {
            log.debug("Interaction hooks for callback interaction between caller:" + caller
                    + ", and callee: " + callee
                    + ". \n\tProvided call back value is: " + provided);
        }

        Collection<DialogInteraction> interactionHook = getInteractionHook(
                caller, callee);

        for (DialogInteraction dialogInteraction : interactionHook) {
            provided = dialogInteraction.getActionReturnValue(provided, caller,
                    callee);
        }

        if (log.isTraceEnabled()) {
            log.trace("After running provided call back value through hooks, the value is: "
                    + provided);
        }

        return provided;
    }

    /**
     * Factory method, creates a DialogInteractionBroker.
     *
     * @return DialogInteractionBroker
     */
    public static DialogInteractionBroker getInstance() {
        DialogInteractionBroker retVal = ServiceLoader
                .<DialogInteractionBroker> getService(DialogInteractionBroker.class);
        return retVal;
    }

    /**
     * Get the collection of interaction hooks for the specified caller.
     *
     * @param caller
     *            the dialog that performs the <em>open</em> action
     * @param callee
     *            the <em>opened</em> dialog
     * @return the dialog interaction hooks to use with specified caller and
     *         callee
     */
    private Collection<DialogInteraction> getInteractionHook(DialogController caller, DialogConstant callee) {

        if (log.isDebugEnabled()) {
            log.debug("Getting interaction hooks for caller: " + caller + " and callee: " + callee);
        }

        HookProviderImpl hookProvider = HookProviderImpl.getInstance();
        List<DialogInteraction> listCaller = hookProvider.getCallerInteractionHook(getCaller(caller));
        List<DialogInteraction> listCallee = hookProvider.getCalleeInteractionHook(callee);
        SetView<DialogInteraction> setFinal= Sets.intersection(Sets.newHashSet(listCaller), Sets.newHashSet(listCallee));
        Collection<DialogInteraction> listFinal= Lists.newArrayList(setFinal);

        if (log.isTraceEnabled()) {
            log.trace("Got interaction hooks: " + listFinal);
        }

        return listFinal;
    }

    private DialogConstant getCaller(DialogController caller) {
        return caller != null ? caller.getDialogConstant() : null;
    }

}
