/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.io.ByteArrayOutputStream;

import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.action.ActionTask;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.view.Resource;
import no.g9.client.core.view.ResourceStore;
import no.g9.service.print.ExportService;
import no.g9.support.ClientContext;
import no.g9.support.ObjectSelection;
import no.g9.support.xml.XmlConverter;

import org.w3c.dom.Document;

/**
 * The export action exports an object selection to a pdf file by using the
 * export service for the specified target dialog.
 * 
 */
final class ExportAction extends ActionTask<Resource> {

    private static final Logger log = Logger.getLogger(ExportAction.class);
    private static final String PDF = ".pdf";
    private static final String MIME_TYPE = "application/pdf";
    private static final boolean PURGE = true;

    // Initialized in constructor
    private final DialogConstant targetDialog;
    private final ByteArrayOutputStream pdfBytes;

    private final XmlConverter xmlConverter = ServiceLoader
            .getService(XmlConverter.class);
    private final ExportService exportService = ServiceLoader
            .getService(ExportService.class);
    private final ResourceStore resourceStore = ServiceLoader
            .getService(ResourceStore.class);
    private final ApplicationController appCtrl;

    /**
     * Create a new export action.
     * 
     * @param target dialog defining the jasper target
     * @param appCtrl application controller
     */
    ExportAction(DialogConstant target, ApplicationController appCtrl) {

        targetDialog = target;
        this.appCtrl = appCtrl;
        pdfBytes = new ByteArrayOutputStream();
    }

    @Override
    public Resource call() throws Exception {
        Object[] taskObject = (Object[]) getTaskObject();
        ObjectSelection os = (ObjectSelection) taskObject[0];
        ClientContext ctx = (ClientContext) taskObject[1];
        if (log.isDebugEnabled()) {
            log.debug("Exporting " + this + " to pdf file");
        }
        Document data = xmlConverter.convert(os, ctx);
        if (log.isTraceEnabled()) {
            log.trace("Converted to xml");
        }
        exportService.exportToPDF(data, targetDialog.getG9Name(), pdfBytes);
        if (log.isTraceEnabled()) {
            log.trace("Exported to pdf");
        }

        Resource pdfResource =
                new Resource(targetDialog.getG9Name() + PDF, MIME_TYPE,
                        pdfBytes.toByteArray());
        resourceStore.putResource(pdfResource);
        appCtrl.getApplicationView().openResource(pdfResource.getId(), PURGE);
        return pdfResource;
    }

    @Override
    public String toString() {
        return "ExportAction [targetDialog=" + targetDialog + "]";
    }

}
