/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import no.esito.log.Logger;
import no.g9.client.core.action.ActionTask;
import no.g9.client.core.controller.DialogController;
import no.g9.service.JGrapeService;
import no.g9.support.ActionType;
import no.g9.support.ClientContext;
import no.g9.support.ObjectSelection;

/**
 * Performs one of the grape actions (FIND, FINDALL, INSERT, UPDATE, DELETE). If
 * the action is either SAVE, INSERT, UPDATE or DELETE, the result is merged (using
 * the dialog controller's merge) before returning.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
final class GrapeActionTask extends ActionTask<Object> {

    /** The logger */
    private final static Logger log = Logger.getLogger(GrapeActionTask.class);

    /** The dialog controller */
    private final DialogController dialogController;

    /** The action type */
    private final ActionType action;


    /**
     * Creates a new grape action of the specified {@link ActionType}.
     *
     * @param dialogController the dialog controller wanting to execute the
     *            action.
     * @param action the action type to execute.
     */
    GrapeActionTask(DialogController dialogController, ActionType action) {
        this.dialogController = dialogController;
        this.action = action;
    }

    @Override
    public Object call() throws Exception {
        Object[] taskObject = (Object[]) getTaskObject();
        final ObjectSelection os = (ObjectSelection) taskObject[0];
        final ClientContext ctx = (ClientContext) taskObject[1];

        if (log.isDebugEnabled()) {
            log.debug("Invoking " + action + ".");
            log.debug("Object selection target is: " + os.getTargetRoleName());
            log.debug("Object selection root is: "
                    + os.getRootObject(os.getTargetRoleName()));
        }

        Object result = invokeService(os, ctx);

        setTaskObject(result);

        if (log.isDebugEnabled()) {
            log.debug("Result from action " + action + " is " + result);
        }



        return result;
    }


    private Object invokeService(ObjectSelection os, ClientContext ctx) {
        JGrapeService service = dialogController.getApplicationController()
                .getServiceProxy();

        switch (action) {
            case FIND:
                return service.find(os, ctx);

            case FINDALL:
                return service.findAll(os, ctx);

            case INSERT:
                if (os.getActions() != null) {
                    return service.save(os, ctx);
                }
                return service.insert(os, ctx);

            case UPDATE:
                if (os.getActions() != null) {
                    return service.save(os, ctx);
                }
                return service.update(os, ctx);

            case SAVE:
                return service.save(os, ctx);

            case DELETE:
                if (os.getActions() != null) {
                    return service.save(os, ctx);
                }
                return service.delete(os, ctx);

            default:
                return null;
        }
    }

}
