/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import no.esito.jvine.action.ActionQueue;
import no.esito.jvine.action.BlockingGuiActionQueue;
import no.g9.client.core.controller.ApplicationController;

/**
 * The jVine application controller
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public abstract class JVineApplicationController {

    private ActionQueue actionQueue = new BlockingGuiActionQueue();

    /** Flag used to control whether to invoke hooks or not. */
    private boolean invokeHooks = true;

    /** The application controller for this thread. Inherited to child threads. */
    private static final InheritableThreadLocal<ApplicationController> CURRENT_APP_CTRL = new InheritableThreadLocal<ApplicationController>();

    private static boolean saveCurrentAppController = false;

    /**
     * Gets the application's action queue.
     * 
     * @return the action queue.
     */
    ActionQueue getActionQueue() {
        return actionQueue;
    }

    /**
     * Get the invokeHooks flag used to determine whether actions should invoke
     * registered hooks or not. If this flag is set to true, action hooks are
     * invoked, otherwise action hooks are ignored.
     * 
     * @return <code>true</code> if action hooks should be invoked.
     */
    final boolean isInvokeHooksOn() {
        return invokeHooks;
    }

    /**
     * Set the invoke hooks flag used to determine whether actions should invoke
     * registered hooks or not. 
     * @param invokeHooks the invoke hooks property.
     * @see #isInvokeHooksOn()
     */
    final void setInvokeHooks(boolean invokeHooks) {
        this.invokeHooks = invokeHooks;
    }


	/**
	 * If set to true, the current application controller will be set in a
	 * InheritableThreadLocal, and thus inherited in all sub threads. If
	 * false, the current application controller will not be set.
	 *
	 * @param saveCurrentAppController the saveCurrentAppController to set
	 */
	public static void setSaveCurrentAppController(boolean saveCurrentAppController) {
		JVineApplicationController.saveCurrentAppController = saveCurrentAppController;
	}

    /**
     * Get the current application controller for this thread.
     *
     * @return the current application controller
     */
    public static ApplicationController getCurrentApplicationController() {
    	return CURRENT_APP_CTRL.get();
    }

    /**
     * Set the current application controller for this thread.
     *
     * @param ctrl the application controller to set
     */
    public static final void setCurrentApplicationController(ApplicationController ctrl) {
    	if (saveCurrentAppController) {
    		CURRENT_APP_CTRL.set(ctrl);
    	}
    }

}
