/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import no.esito.jvine.action.ActionMethod;
import no.esito.jvine.action.ActionMethodRunner;
import no.esito.jvine.action.ThreadManager;
import no.esito.jvine.model.TreeNode;
import no.g9.client.core.action.ActionFactory;
import no.g9.client.core.action.ActionTask;
import no.g9.client.core.action.EventContext;
import no.g9.client.core.action.G9Action;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.message.BlockingMessageCallback;
import no.g9.client.core.validator.ValidationResult;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ViewModel;
import no.g9.exception.G9BaseException;
import no.g9.exception.G9ClientException;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.ObjectSelection;
import no.g9.support.action.ActionTarget;
import no.g9.support.convert.ConvertException;

/**
 * This is the "privileged" part of the dialog controller, which delegates
 * all methods to the actual dialog controller's package-visibility methods.
 * Instances of this class are intended for create-and-throw away use --- create
 * an instance when needed and throw it away when you don't want it any more.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public abstract class JVineController implements JVineDialogController {

    private StateController delegate;



    /**
     * Get the invokeHooks flag used to determine whether actions should invoke
     * registered hooks or not. If this flag is set to true, action hooks are
     * invoked, otherwise action hooks are ignored.
     *
     * @return <code>true</code> if action hooks should be invoked.
     */
    public boolean isInvokeHooksOn() {
        return delegate.isInvokeHooksOn();
    }

    /**
     * Set the invoke hooks flag used to determine whether actions running from
     * this controller should invoke registered hooks or not.
     * @param invokeHooks the invoke hooks property.
     * @see #isInvokeHooksOn()
     */
    public void setInvokeHooks(boolean invokeHooks) {
        delegate.setInvokeHooks(invokeHooks);
    }

    /**
     * Get a JVineController instance that gives access to the "privileged" part
     * of the specified dialog controller.
     * @param delegate the dialog controller to wrap with this jVine Controller
     * @return the JVineController for the specified dialog controller.
     */
    public static final JVineController getInstance(DialogController delegate) {
        return new JVineController(delegate) {
            // Empty.
        };
    }

    private JVineController(DialogController delegate) {
        this.delegate = delegate;
    }

    @Override
    public void addConverterException(ConvertException converterException) {
        delegate.addConverterException(converterException);
    }

    /**
     * Gets the dialog view class of this dialog.
     *
     * @param <T> The dialog view type
     * @return the dialog view.
     */
    public final <T extends DialogView> T getDialogView() {
        return delegate.<T>getDialogViewInternal();
    }

    @Override
    public void addOSNode(OSNode<?> node) {
        delegate.addOSNode(node);
    }

    @Override
    public void addValidationResult(ValidationResult validationResult) {
        delegate.addValidationResult(validationResult);
    }

    @Override
    public boolean checkClose() {
        return delegate.checkClose();
    }

    @Override
    public void checkValidationAndConvert(boolean isObtain)
            throws G9ClientException {
        delegate.checkValidationAndConvert(isObtain);
    }

    @Override
    public void clearCurrent(RoleConstant role) {
        delegate.clearCurrent(role);
    }

    @Override
    public void closeDialogController() {
        delegate.closeDialogController();
    }

    @Override
    public void dispatch(EventContext event) {
        delegate.dispatch(event);
    }

    @Override
    public <T> G9Action<T> getAction(ActionType actionType, ActionTarget target,
            Class<T> targetType, ActionTask<T> actionTask) {
        return delegate.getAction(actionType, target, targetType, actionTask);
    }

    @Override
    public ActionFactory getActionFactory() {
        return delegate.getActionFactory();
    }

    @Override
    public ActionMethodRunner getActionMethodRunner() {
        return delegate.getActionMethodRunner();
    }

    @Override
    public <T> Collection<T> getAllInstances(OSNode<T> role) {
        return delegate.getAllInstances(role);
    }

    @Override
    public ActionTask<Boolean> getCheckDialogTask(ActionType checkType,
            ActionTarget target) {
        return delegate.getCheckDialogTask(checkType, target);
    }

    @Override
    public ActionTask<Void> getDisplayDialogTask(ActionTarget target, ActionType actionType) {
        return delegate.getDisplayDialogTask(target, actionType);
    }

    @Override
    public FieldData getFieldData(RoleConstant roleConstant) {
        return delegate.getFieldData(roleConstant);
    }

    @Override
    public BlockingMessageCallback getMessageCallback() {
        return delegate.getMessageCallback();
    }

    @Override
    public <T> OSNode<T> getOSNode(RoleConstant role) {
        return delegate.getOSNode(role);
    }

    @Override
    public ActionTask<ObjectSelection> getObtainDialogTask(ActionTarget target, ActionType actionType) {
        return delegate.getObtainDialogTask(target, actionType);
    }

    @Override
    public Collection<OSNode<?>> getOneRelatedChildren(RoleConstant role) {
        return delegate.getOneRelatedChildren(role);
    }

    @Override
    public OSNode<?>[] getRootNodes() {
        return delegate.getRootNodes();
    }

    @Override
    public Map<RoleConstant, OSNode<?>> getRoots() {
        return delegate.getRoots();
    }

    @Override
    public TreeNode<?> getSentinel() {
        return delegate.getSentinel();
    }

    @Override
    public List<ValidationResult> getValidationResult() {
        return delegate.getValidationResult();
    }

    @Override
    public ViewModel getViewModel() {
        return delegate.getViewModel();
    }

    @Override
    public boolean isCurrent(Object inst, OSNode<?> role) {
        return delegate.isCurrent(inst, role);
    }

    @Override
    public void registerEventMethod(String eventMethodName,
            ActionMethod eventMethodInvoker) {
        delegate.registerEventMethod(eventMethodName, eventMethodInvoker);
    }

    @Override
    public void registerView(DialogView dialogView) {
        delegate.registerView(dialogView);
    }

    @Override
    public void reportToView(Collection<OSNode<?>> changedNodes,
            Collection<OSNode<?>> includeNodes, OSNode<?> excludeNode) {
        delegate.reportToView(changedNodes, includeNodes, excludeNode);
    }

    @Override
    public void setActionMethodRunner(ActionMethodRunner actionMethodRunner) {
        delegate.setActionMethodRunner(actionMethodRunner);
    }

    @Override
    public void setCurrent(FieldData instanceFieldData) {
        delegate.setCurrent(instanceFieldData);
    }

    @Override
    public DialogKey getDialogKey() {
        return delegate.getDialogKey();
    }

    /**
     * Releases the gui thread
     */
    public void releaseGui() {
        getActionMethodRunner().releaseGui();
    }

    /**
     * Invoke the specified runnable on the gui thread.
     * @param guiMethod the runnable to be run by the gui thread.
     * @throws InvocationTargetException exceptions thrown by the runnable are
     * wrapped in an InvocationTargetException and re-thrown.
     */
    public void invokeOnGui(Runnable guiMethod) throws InvocationTargetException {
        if (ThreadManager.isWorkerThread()) {
            getActionMethodRunner().invokeOnGui(guiMethod);
        } else {
            guiMethod.run();
        }
    }

    /**
     * Invoke the specified callable on the gui thread.
     * @param <V> the return value
     * @param guiMethod the callable to be invoked by the gui thread.
     * @return the callable's return value
     * @throws InvocationTargetException exceptions thrown by the callable are
     * wrapped in an InvocationTargetException and re-thrown.
     */
    public <V> V invokeOnGui(Callable<V> guiMethod) throws InvocationTargetException {
        if (ThreadManager.isWorkerThread()) {
            return getActionMethodRunner().invokeOnGui(guiMethod);
        }
        try {
            return guiMethod.call();
        } catch (Exception e) {
            throw new G9BaseException(e);
        }
    }

    /**
     * Set the caller that opened this dialog.
     * @param dialogController the dialog controller that opened this dialog
     */
    public void setCaller(DialogController dialogController) {
        delegate.setCaller(dialogController);

    }

    /**
     * Get the caller that opened this dialog.
     * @return the dialog controller that opened this dialog
     */
    public DialogController getCaller() {
        return delegate.getCaller();
    }




}
