/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import no.esito.jvine.rpc.ActualParameter;
import no.g9.client.core.action.Parameter;
import no.g9.client.core.action.ParameterBinding;
import no.g9.client.core.action.RemoteServiceTarget;
import no.g9.client.core.controller.DialogController;
import no.g9.os.OSRole;

/**
 * Utility class.
 * 
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public class RoleLocator {

    /**
     * Get the nodes applicable for check change. 
     * @param dialogController the dialog controller of the object selection
     * @param target the target containing parameter bindings with nodes
     * @return an array of nodes that should be checked.
     */
    public static OSNode<?>[] getNodesToCheck(DialogController dialogController, RemoteServiceTarget target) {
        Collection<OSNode<?>> extractNodes = extractNodes(dialogController, target);
        List<OSNode<?>> topRoles = getTopRoles(dialogController, extractNodes);
        return topRoles.toArray(new OSNode<?>[0]);
    }

    private static Collection<OSNode<?>> extractNodes(DialogController dialogController,
            RemoteServiceTarget target) {
        List<OSNode<?>> nodes = new ArrayList<OSNode<?>>();
        JVineController ctrl = JVineController.getInstance(dialogController);
        List<ParameterBinding<?>> parameterBindings = target.getParameterBindings();
        for (ParameterBinding<?> parameterBinding : parameterBindings) {
            fillNodeList(nodes, ctrl, parameterBinding);
        }
        ParameterBinding<?> returnParameter = target.getReturnParameter();
        
        fillNodeList(nodes, ctrl, returnParameter);
        
        return nodes;
    }

    private static void fillNodeList(List<OSNode<?>> nodes,
            JVineController ctrl, ParameterBinding<?> parameterBinding) {
        Parameter<?> actualParameter = parameterBinding.getActualParameter();
        if (actualParameter instanceof ActualParameter<?>) {
            ActualParameter<?> ap = (ActualParameter<?>) actualParameter;
            if (ap.getRole() != null) {
                nodes.add(ctrl.getOSNode(ap.getRole()));
            }
        }
    }
    
    /**
     * Get the top roles from the collection of roles.
     * @param dialogController the dialog controller
     * @param roles the collection of roles to extract top roles from.
     * @return the minimum set of roles that spans the all roles in the collection.
     */
    private static List<OSNode<?>> getTopRoles(DialogController dialogController, Collection<OSNode<?>> roles) {
        JVineController ctrl = JVineController.getInstance(dialogController);
        Set<OSNode<?>> allRoles = new HashSet<OSNode<?>>();
        allRoles.addAll(roles);
        Set<OSNode<?>> roots = new HashSet<OSNode<?>>();
        for (OSNode<?> osNode : roles) {
            OSRole<?> root = osNode.getRoot();
            roots.add(ctrl.<Object>getOSNode(root.getRoleConstant()));
        }
        ArrayList<OSNode<?>> found = new ArrayList<OSNode<?>>();
        for (OSNode<?> osRole : roots) {
            found.addAll(locateRole(allRoles, osRole));
        }
        
        return found;
        
    }
    
    private static List<OSNode<?>> locateRole(Set<OSNode<?>> allRoles, OSNode<?> node) {
        ArrayList<OSNode<?>> found = new ArrayList<OSNode<?>>();
        if (allRoles.contains(node)) {
            found.add(node);
            return found;
        }

        Collection<OSNode<?>> children = node.getOSNodeChildren();
        for (OSNode<?> osNode : children) {
            List<OSNode<?>> locateRole = locateRole(allRoles, osNode);
            found.addAll(locateRole);
        }
        return found;
    }
}
