/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.io.ByteArrayOutputStream;

import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.action.ActionTask;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.view.Resource;
import no.g9.client.core.view.ResourceStore;
import no.g9.client.spreadsheet.SpreadsheetExporter;
import no.g9.client.spreadsheet.SpreadsheetService;
import no.g9.support.ClientContext;
import no.g9.support.ObjectSelection;

/**
 * The spreadsheet action exports an object selection to a spreadsheet by using
 * a generated spreadsheet exporter for the specified target dialog.
 *
 * The spreadsheet is stored in the ResourceStore of the application.
 *
 */
final class SpreadsheetAction extends ActionTask<Resource> {

    private static final Logger log = Logger.getLogger(SpreadsheetAction.class);
    private static final boolean PURGE = true;

    // Initialized in constructor
    private final DialogConstant targetDialog;
    private final ByteArrayOutputStream spreadsheetBytes;

    private final SpreadsheetService spreadsheetService = ServiceLoader.getService(SpreadsheetService.class);
    private final ResourceStore resourceStore = ServiceLoader.getService(ResourceStore.class);
    private final ApplicationController appCtrl;

    /**
     * Create a new spreadsheet action.
     *
     * @param target dialog defining the spreadsheet target
     * @param appCtrl application controller
     */
    SpreadsheetAction(DialogConstant target, ApplicationController appCtrl) {

        targetDialog = target;
        this.appCtrl = appCtrl;
        spreadsheetBytes = new ByteArrayOutputStream();
    }

    @Override
    public Resource call() throws Exception {
        Object[] taskObject = (Object[]) getTaskObject();
        ObjectSelection os = (ObjectSelection) taskObject[0];
        ClientContext ctx = (ClientContext) taskObject[1];
        if (log.isDebugEnabled()) {
            log.debug("Exporting " + this + " as a spreadsheet resource");
        }
        SpreadsheetExporter exporter = spreadsheetService.exportToSpreadsheet(os, targetDialog, spreadsheetBytes, ctx);
        DialogController dc = appCtrl.getDialogController(targetDialog);
        if (dc != null) {
        	dc.setSpreadsheetExporter(exporter);
        }
        Resource xlResource = new Resource(targetDialog.getG9Name() + spreadsheetService.getFileSuffix(), spreadsheetService.getMimeType(), spreadsheetBytes.toByteArray());
        resourceStore.putResource(xlResource);
        appCtrl.getApplicationView().openResource(xlResource.getId(), PURGE);
        return xlResource;
    }

    @Override
    public String toString() {
        return "SpreadsheetAction [targetDialog=" + targetDialog + "]";
    }

}
