/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.model;

import java.util.Collection;

import no.esito.jvine.controller.OSNode;

/**
 * Representation of a tree node of the displayable instances.
 * 
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 * @param <T> The domain type of this tree node.
 */
public interface TreeNode<T> {

    /**
     * Get the parent tree node.
     * 
     * @return the parent tree node.
     */
    public TreeNode<?> getParent();

    /**
     * Get the collection of child tree nodes.
     * 
     * @return the child tree nodes.
     */
    public Collection<? extends TreeNode<?>> getChildren();

    /**
     * Adds a tree node child to this tree node.
     * 
     * @param child the tree node to add.
     */
    public void addChild(TreeNode<?> child);

    /**
     * Get the object selection node of this tree node.
     * 
     * @return the object selection node of this tree node.
     */
    public OSNode<T> getOSNode();

    /**
     * Set the current instance of this tree node. This operation will propagate
     * downwards in the tree, effecting all nodes below.
     * 
     * @param current the new current instance.
     * @return a collection of the nodes that changed as part of the current
     *         changing.
     */
    public Collection<OSNode<?>> setCurrentInstance(T current);
    
    /**
     * Set the current instances of this tree node. <em>NOTE:</em> this is only 
     * applicable to root nodes.
     * @param instances the collection of (root-role) instances.
     * @return a collection of the nodes that changed as part of this operation.
     */
    public Collection<OSNode<?>> setCurrentRootInstances(Collection<T> instances);


    /**
     * Get the current instance of this tree node.
     * 
     * @return the current instance.
     */
    public T getCurrentInstance();

    /**
     * Get the collection of displayable instances. This is determined based on
     * the parent node and the current parent instance.
     * 
     * @return the instances that can be reached by traversing the object
     *         (instance) tree from the root and downwards to this node.
     */
    public Collection<T> getInstances();

    /**
     * Get the relation from the current instance to the specified child role.
     * 
     * @param child the object selection node of the child
     * @return the relation from this current instance to the child.
     */
    public Object getChildRelation(OSNode<?> child);

    /**
     * Select the current instance based on the current selected parent
     * instance.
     * 
     * @return the collection of nodes that should be updated as part of this
     *         computation.
     */
    public Collection<OSNode<?>> computeCurrent();

    /**
     * Removes all instances and propagates the change down in the tree.
     * 
     * @param intercept if true, clear interceptor will be invoked.
     * @return the collection of affected nodes.
     */
    public Collection<OSNode<?>> clear(boolean intercept);
    
    /**
     * Removes the current instance and propagates the change down in the tree.
     *  @param intercept if true, clear interceptor will be invoked.
     * @return the collection of affected nodes.
     */
    public Collection<OSNode<?>> clearCurrent(boolean intercept);

    /**
     * Removes the current instance without propagating the change to children.
     * 
     * @return the collection of changed nodes (viz. this node).
     */
    public Collection<OSNode<?>> removeCurrent();

}