/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.model;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;

import no.esito.jvine.controller.OSNode;
import no.esito.log.Logger;

/**
 * Acts as a sentinel that holds all root instances. The instances are mapped
 * from the corresponding OSNode.
 *
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 *
 */
public class TreeNodeSentinel extends TreeNodeImpl<Object> {

    private static final Logger log = Logger.getLogger(TreeNodeSentinel.class);

    /** Map from osNode to collection of instances */
    private final Map<OSNode<?>, Collection<?>> childInstanceMap;

    /**
     * The name of the guarded object selection.
     */
    private final Enum<?> objectSelectionName;

    /**
     * Constructs a new tree node sentinel.
     *
     * @param objectSelectionName the (generated) enum denoting the object
     *            selection.
     */
    public TreeNodeSentinel(Enum<?> objectSelectionName) {
        super(null, null);
        childInstanceMap = new HashMap<OSNode<?>, Collection<?>>();
        this.objectSelectionName = objectSelectionName;
    }

    /**
     * Constructs a new tree node sentinel without any object selection.
     */
    public TreeNodeSentinel() {
        this(null);
    }

    /**
     * Add a collection of root instances to this sentinel.
     *
     * @param osNode the role of the root instances to add.
     * @param instances the collection of instances to add.
     */
    void addRootInstances(OSNode<?> osNode, Collection<?> instances) {
        // ensure that instances is kept in a set
        LinkedHashSet<Object> tmpSet = new LinkedHashSet<Object>();
        tmpSet.addAll(instances);
        childInstanceMap.put(osNode, tmpSet);
        if (log.isTraceEnabled()) {
            log.trace(this + " adding instances for " + osNode);
        }
    }



    @Override
    public Object getChildRelation(OSNode<?> child) {
        return childInstanceMap.get(child);
    }

    @Override
    public Object getCurrentInstance() {
        String msg = this + " cannot get current from sentinel.";
        throw new UnsupportedOperationException(msg);
    }

    @Override
    public Collection<Object> getInstances() {
        String msg = this + " cannot get instances from sentinel.";
        throw new UnsupportedOperationException(msg);
    }

    @Override
    public OSNode<Object> getOSNode() {
        String msg = this + " cannot get os node from sentinel.";
        throw new UnsupportedOperationException(msg);
    }

    @Override
    public TreeNode<?> getParent() {
        return null;
    }

    @Override
    public Collection<OSNode<?>> setCurrentInstance(Object current) {
        String msg = this + " cannot set current on sentinel.";
        throw new UnsupportedOperationException(msg);
    }

    /**
     * Clears the specified root role.
     *
     * @param rootRole the root role to clear
     */
    void clearRootInstance(OSNode<?> rootRole) {
        if (log.isTraceEnabled()) {
            log.trace(this + " Removing instances of " + rootRole);
        }
        childInstanceMap.remove(rootRole);
    }

    @Override
    public String toString() {
        return "TreeNodeSentinel [" + objectSelectionName + "]";
    }

}
