/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.rpc;

import no.esito.log.Logger;
import no.g9.client.core.action.ParameterBinding;
import no.g9.support.convert.ConvertException;
import no.g9.support.convert.Converter;

/**
 * An implementation of the ParameterBinding interface.
 *
 * @param <A> the actual parameter type.
 */
public class ParameterBindingImpl<A> implements ParameterBinding<A> {

    private final Converter<A, ?> converter;

    private final ActualParameter<A> actualParameter;

    private final boolean isReturnValue;

    private final String formalParameterName;

    private final static Logger log = Logger.getLogger(ParameterBinding.class);

    /**
     * Convenience method. Creates a new parameter binding (possibly a return
     * type) of the specified type and returns it.
     *
     * @param <A> type of actual parameter
     * @param formalParameterName The name declared name of the formal parameter
     * @param actualParameter the actual parameter
     * @param converter the converter
     * @param isReturnValue is this a return value?
     * @return a parameter binding
     */
    public static <A> ParameterBinding<A> getBinding(String formalParameterName,
        ActualParameter<A> actualParameter, Converter<A, ?> converter, boolean isReturnValue) {
        return new ParameterBindingImpl<A>(formalParameterName, actualParameter, converter, isReturnValue);
    }

    /**
     * Convenience method. Creates a new parameter binding of the specified type
     * and returns it.
     *
     * @param <A> The type of parameter binding
     * @param formalParameterName the name of the formal parameter
     * @param actualParameter the name of the actual parameter
     * @param converter the converter used for converting from actual to formal
     *            parameter type (may bee null).
     * @return a parameter binding.
     */
    public static <A> ParameterBinding<A> getBinding(String formalParameterName,
        ActualParameter<A> actualParameter, Converter<A, ?> converter) {
        return new ParameterBindingImpl<A>(formalParameterName, actualParameter, converter, false);
    }

    /**
     * Constructs a new service parameter with the specified name and converter.
     *
     * @param formalParameterName The declared name of the formal parameter.
     * @param actualParameter the actual parameter
     * @param converter the converter used between actual and formal parameter.
     * @param isReturnValue is this a return value?
     * @throws NullPointerException if the formal or actual parameter name is
     *             null.
     */
    public ParameterBindingImpl(String formalParameterName, ActualParameter<A> actualParameter,
        Converter<A, ?> converter, boolean isReturnValue) {
        this.formalParameterName = formalParameterName;
        this.actualParameter = actualParameter;
        this.converter = converter;
        this.isReturnValue = isReturnValue;
    }

    @Override
    public final Converter<A, ?> getConverter() {
        return converter;
    }

    @Override
    public final boolean hasConverter() {
        return converter != null;
    }

    @Override
    public ActualParameter<A> getActualParameter() {
        return actualParameter;
    }

    @Override
    public boolean isReturnValue() {
        return isReturnValue;
    }

    @Override
    public String getFormalParameterName() {
        return formalParameterName;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <F> F getFormalParameterValue() throws ConvertException {
        A actualParameterValue = getActualParameter().getParameterValue();
        if (hasConverter()) {
            Converter<A, F> conv = (Converter<A, F>) getConverter();
            return conv.fromModel(actualParameterValue, null);
        }
        return (F) actualParameterValue;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <F> A getActualParameterValue(F formalParameterValue) throws ConvertException {
        log.debug("Getting actual parameter value from formal value: " + formalParameterValue + " (class: "
            + formalParameterValue.getClass() + ")");
        if (hasConverter()) {

            Converter<A, F> conv = (Converter<A, F>) getConverter();
            log.debug("Using " + conv + " to convert formal value.");
            return conv.toModel(formalParameterValue, null);
        }
        log.debug("No converter - attemting a cast.");
        return (A) formalParameterValue;
    }

    @Override
    public A obtainParameterValue() {
        if (!getActualParameter().isObtained()) {
            getActualParameter().setParameterValue(getActualParameter().obtainValue());
        }
        return getActualParameter().getParameterValue();
    }

    @Override
    public String toString() {
        return "ParameterBindingImpl ["
            + (actualParameter != null ? "actualParameter=" + actualParameter + ", " : "")
            + (converter != null ? "converter=" + converter + ", " : "")
            + (formalParameterName != null ? "formalParameterName=" + formalParameterName + ", " : "")
            + "isReturnValue=" + isReturnValue + "]";
    }

    @Override
    public void setActualParameterValue(A actualParameterValue) {
        getActualParameter().setParameterValue(actualParameterValue);

    }

    @Override
    public <F> void setFormalParameterValue(F formalParameterValue) throws ConvertException {
        A actualParameterValue = null;
        if (formalParameterValue != null) {
            actualParameterValue = getActualParameterValue(formalParameterValue);
        }
        setActualParameterValue(actualParameterValue);
    }

}
