/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.view;

import java.util.HashMap;
import java.util.Map;

import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.ViewConstant;
import no.g9.client.core.view.BooleanProperty;
import no.g9.client.core.view.EffectProperty;
import no.g9.client.core.view.Property;
import no.g9.client.core.view.PropertyManager;
import no.g9.client.core.view.StringProperty;

/**
 * Manages properties that can be applied to a dialog object.
 */
public final class DefaultPropertyManager implements PropertyManager {

	private final Map<DialogObjectConstant, Map<Property<?>, Object>> dialogObjectProperties = new HashMap<DialogObjectConstant, Map<Property<?>, Object>>();

	@Override
	public final <T> void setProperty(DialogObjectConstant dialogObject,
			Property<T> property, T value) {
		Map<Property<?>, Object> propMap = dialogObjectProperties
				.get(dialogObject);
		if (propMap == null) {
			propMap = new HashMap<Property<?>, Object>();
			dialogObjectProperties.put(dialogObject, propMap);
		}

		propMap.put(property, value);

	}

	/**
	 * Set the specified String property
	 * 
	 * @param dialogObject
	 *            the dialog object whose property to set.
	 * @param property
	 *            the property to set
	 * @param value
	 *            the property value.
	 */
	public final void setProperty(DialogObjectConstant dialogObject,
			String property, Object value) {
		StringProperty stringProperty = StringProperty.getInstance(
				dialogObject, property);
		setProperty(dialogObject, stringProperty, value);
	}

    @Override
    @SuppressWarnings("unchecked")
    public final <T> T getProperty(DialogObjectConstant dialogObject, Property<T> property) {
	    if (dialogObject != null && isRecursiveBooleanProperty(property)) {
	        return (T) getRecursiveBooleanProperty(dialogObject, (Property<Boolean>) property);
	    }
	    return getPropertyInternal(dialogObject, property);
	}

	@SuppressWarnings("unchecked")
    private <T> T getPropertyInternal(DialogObjectConstant dialogObject, Property<T> property) {
        Map<Property<?>, Object> propMap = dialogObjectProperties.get(dialogObject);
        if (propMap != null) {
            Object object = propMap.get(property);
            if (object != null) {
                return (T) object;
            }
        }
        return property.getDefaultValue();
    }

	private boolean isRecursiveBooleanProperty(Property<?> property) {
	    if (property == BooleanProperty.ENABLED || property == BooleanProperty.SHOWN) {
	        return true;
	    }
	    return false;
	}
	
    private Boolean getRecursiveBooleanProperty(DialogObjectConstant dialogObject, Property<Boolean> property) {
	    Boolean value = getPropertyInternal(dialogObject, property);
        DialogObjectConstant parent = getParent(dialogObject);
        while (parent != null && value) {
            value = getPropertyInternal(parent, property);
            parent = getParent(parent);
        }
	    return value;
	}

    private DialogObjectConstant getParent(DialogObjectConstant dialogObject) {
        ViewConstant parent = dialogObject.getParent();
        if (parent instanceof DialogObjectConstant) {
            return (DialogObjectConstant) parent;
        }
        return null;
    }

    /**
	 * Get the specified string property value. If the property is not set,
	 * <code>null</code> is returned (use
	 * {@link #isPropertySet(DialogObjectConstant, Property)} to test if a
	 * property is set.
	 * 
	 * @param dialogObject
	 *            the dialog object whose property to get.
	 * @param property
	 *            the property to get.
	 * @return the property value.
	 * @throws ClassCastException
	 *             if the actual property type does not match the expected
	 *             property value type.
	 */
	public final Object getProperty(DialogObjectConstant dialogObject,
			String property) {
		StringProperty stringProperty = StringProperty.getInstance(
				dialogObject, property);
		return getProperty(dialogObject, stringProperty);
	}

	@Override
	public final boolean isPropertySet(DialogObjectConstant dialogObject,
			Property<?> property) {
		Map<Property<?>, Object> propMap = dialogObjectProperties
				.get(dialogObject);
		if (propMap != null) {
			return propMap.containsKey(property);
		}
		return false;
	}

	/**
	 * Test a string property is set.
	 * 
	 * @param dialogObject
	 *            the dialog object that possibly has the property set.
	 * @param property
	 *            the property
	 * @return <code>true</code> if the property is set, otherwise
	 *         <code>false</code>.
	 */
	public final boolean isPropertySet(DialogObjectConstant dialogObject,
			String property) {
		StringProperty stringProperty = StringProperty.getInstance(
				dialogObject, property);
		return isPropertySet(dialogObject, stringProperty);
	}

	@Override
	public final void clearAll() {
		dialogObjectProperties.clear();
	}

	@Override
	public final void clearAll(DialogObjectConstant dialogConstant) {
		dialogObjectProperties.remove(dialogConstant);
	}

	@Override
	public final void clear(DialogObjectConstant dialogObject,
			Property<?> property) {
		Map<Property<?>, Object> propMap = dialogObjectProperties
				.get(dialogObject);
		if (propMap != null) {
			propMap.remove(property);
		}
	}

	/**
	 * Set the editable property of the specified dialog object.
	 * 
	 * @param dialogObject
	 *            the dialog object
	 * @param editable
	 *            the value of the editable property.
	 */
	public final void setEditable(DialogObjectConstant dialogObject,
			boolean editable) {
		setProperty(dialogObject, BooleanProperty.EDITABLE, Boolean
				.valueOf(editable));
	}

	/**
	 * Get the value of the specified dialog object's editable property value.
	 * 
	 * @param dialogObject
	 *            the dialog object
	 * @return the value of the editable property.
	 */
	public final boolean isEditable(DialogObjectConstant dialogObject) {
		Boolean property = getProperty(dialogObject, BooleanProperty.EDITABLE);
		return property.booleanValue();
	}

	/**
	 * Set the enabled property of the specified dialog object.
	 * 
	 * @param dialogObject
	 *            the dialog object
	 * @param enabled
	 *            the value of the enabled property.
	 */
	public final void setEnabled(DialogObjectConstant dialogObject,
			boolean enabled) {
		setProperty(dialogObject, BooleanProperty.ENABLED, Boolean
				.valueOf(enabled));
	}

	/**
	 * Get the value of the specified dialog object's enabled property value.
	 * 
	 * @param dialogObject
	 *            the dialog object
	 * @return the value of the enabled property.
	 */
	public final boolean isEnabled(DialogObjectConstant dialogObject) {
		Boolean property = getProperty(dialogObject, BooleanProperty.ENABLED);
		return property.booleanValue();
	}

	/**
	 * Set the mandatory property of the specified dialog object.
	 * 
	 * @param dialogObject
	 *            the dialog object.
	 * @param mandatory
	 *            the value of the mandatory property.
	 */
	public final void setMandatory(DialogObjectConstant dialogObject,
			boolean mandatory) {
		setProperty(dialogObject, BooleanProperty.MANDATORY, Boolean
				.valueOf(mandatory));
	}
	
	/**
	 * Get the mandatory property of the specified dialog object.
	 * @param dialogObject the dialog object
	 * @return the value of the mandatory property
	 */
	public final boolean getMandatory(DialogObjectConstant dialogObject) {
	    return getProperty(dialogObject, BooleanProperty.MANDATORY);
	}

	/**
	 * Get the specified dialog object's mandatory property value.
	 * 
	 * @param dialogObject
	 *            the dialog object.
	 * @return the value of the mandatory property.
	 */
	public final boolean isMandatory(DialogObjectConstant dialogObject) {
		return getProperty(dialogObject, BooleanProperty.MANDATORY)
				.booleanValue();
	}

	/**
	 * Set the shown property of the specified dialog object.
	 * 
	 * @param dialogObject
	 *            the dialog object.
	 * @param shown
	 *            the value of the shown property.
	 */
	public final void setShown(DialogObjectConstant dialogObject, boolean shown) {
		setProperty(dialogObject, BooleanProperty.SHOWN, Boolean.valueOf(shown));
	}

	/**
	 * Get the specified dialog object's expanded property value.
	 * 
	 * @param dialogObject
	 *            the dialog object.
	 * @return the value of the expanded property.
	 */
	public final boolean isExpanded(DialogObjectConstant dialogObject) {
		return getProperty(dialogObject, BooleanProperty.EXPANDED)
				.booleanValue();
	}

	/**
	 * Set the expanded property of the specified dialog object.
	 * 
	 * @param dialogObject
	 *            the dialog object.
	 * @param expanded
	 *            the value of the expanded property.
	 */
	public final void setExpanded(DialogObjectConstant dialogObject,
			boolean expanded) {
		setProperty(dialogObject, BooleanProperty.EXPANDED, Boolean
				.valueOf(expanded));
	}

	/**
	 * Get the specified dialog object's shown property value.
	 * 
	 * @param dialogObject
	 *            the dialog object.
	 * @return the value of the shown property.
	 */
	public final boolean isShown(DialogObjectConstant dialogObject) {
		return getProperty(dialogObject, BooleanProperty.SHOWN).booleanValue();
	}

	   /**
     * Set the effect property of the specified dialog object.
     * 
     * @param dialogObject
     *            the dialog object.
     * @param effect
     *            the value of the effect property.
     */
    public final void setEffect(DialogObjectConstant dialogObject,
            Object effect) {
        setProperty(dialogObject, EffectProperty.STD_EFFECT, effect);
    }

    /**
     * Get the specified dialog object's effect property value.
     * 
     * @param dialogObject
     *            the dialog object.
     * @return the value of the effect property.
     */
    public final Object getEffect(DialogObjectConstant dialogObject) {
        return getProperty(dialogObject, EffectProperty.STD_EFFECT);
    }

	@Override
	public final String listProperties(DialogObjectConstant dialogObject) {
		Map<Property<?>, Object> propMap = dialogObjectProperties
				.get(dialogObject);
		return propMap != null ? propMap.toString() : "<none>";
	}

	@Override
	public final String listProperties() {
		String s = "[";
		for (DialogObjectConstant dialogObject : dialogObjectProperties
				.keySet()) {
			s += dialogObject + ": ";
			s += listProperties(dialogObject) + ", ";
		}
		s = s.substring(0, s.lastIndexOf(','));
		s += "]";
		return s;
	}

	@Override
	public String toString() {
		return DefaultPropertyManager.class.getSimpleName()
				+ ", # properties: " + dialogObjectProperties.size();
	}

}
