/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.util.ArrayList;
import java.util.List;

/**
 * A wrapper class for a List of action hooks. This list holds all the user hooks
 * for a given jVine action.
 *
 * @param <T> The target class the actions in this list operates on
 */
public class ActionHookList<T> {
    
    private List<ActionHook<T>> hookList = new ArrayList<ActionHook<T>>();

    /**
     * @return the list of action hooks
     */
    public List<ActionHook<T>> getHooks() {
        return hookList;
    }

    /**
     * Returns the checkable hooks.
     *
     * @return a list of the checkable hooks, possibly an empty list
     */
    List<Checkable> getCheckableHooks() {
        List<Checkable> hooks = new ArrayList<Checkable>();
        for (ActionHook<?> hook : getHooks()) {
            if (hook instanceof Checkable) {
                hooks.add((Checkable) hook);
            }
        }
        return hooks;
    }

    /**
     * Returns the obtainable hooks.
     *
     * @return a list of the obtainable hooks, possibly an empty list
     */
    List<Obtainable> getObtainableHooks() {
        List<Obtainable> hooks = new ArrayList<Obtainable>();
        for (ActionHook<?> hook : getHooks()) {
            if (hook instanceof Obtainable) {
                hooks.add((Obtainable) hook);
            }
        }
        return hooks;
    }

    /**
     * Returns the displayable hooks.
     *
     * @return a list of the displayable hooks, possibly an empty list
     */
    List<Displayable> getDisplayableHooks() {
        List<Displayable> hooks = new ArrayList<Displayable>();
        for (ActionHook<?> hook : getHooks()) {
            if (hook instanceof Displayable) {
                hooks.add((Displayable) hook);
            }
        }
        return hooks;
    }
    
    /**
     * Set the current G9 action for hooks in the list. If a hook already has a current
     * action (which is running), this method will block until the current
     * action is finished, before setting the new, current action.
     * 
     * @param currentAction the action to associate with this hook
     * @throws InterruptedException if interrupted while blocking.
     */
    protected void setCurrentAction(G9Action<T> currentAction) throws InterruptedException {        
        for (ActionHook<T> hook : hookList) {
            hook.setCurrentAction(currentAction);
        }
    }

    /**
     * Remove the given action from hooks in the list.
     *
     * Calls removeCurrentAction on each hook in the list. This is a blocking method
     * if the given action is not the current action.
     * 
     *
     * @param action The action to remove from the hooks
     * @throws InterruptedException if interrupted while blocking
     */
    protected void removeCurrentAction(G9Action<T> action) throws InterruptedException  {
        for (ActionHook<T> hook : hookList) {
            hook.removeCurrentAction(action);
        }
    }
}
