/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

import no.esito.jvine.action.CancelException;
import no.esito.jvine.action.HookMethod;
import no.esito.jvine.controller.JVineController;
import no.esito.log.Logger;
import no.g9.client.core.controller.DialogController;
import no.g9.support.ActionType;
import no.g9.support.ClientContext;
import no.g9.support.ObjectSelection;
import no.g9.support.action.ActionTarget;

/**
 * @param <V> The action result
 */
@SuppressWarnings({"rawtypes"})
public class ObtainableAction<V> extends CheckableAction<V> {

    /** The obtainable hooks (if any) */
    private List<Obtainable> obtainableHooks = new ArrayList<Obtainable>();

    /** The logger */
    private static Logger log = Logger.getLogger(ObtainableAction.class);

    /**
     * Constructs a new obtainable g9 action.
     *
     * @param actionType the action type
     * @param actionTarget the action target
     * @param actionTask the callable implementing the action task
     * @param resultClass the target class
     * @param dialogController the dialog controller that initiated this action
     */
    protected ObtainableAction(ActionType actionType, ActionTarget actionTarget,
            ActionTask<V> actionTask, Class<V> resultClass,
            DialogController dialogController) {
        super(actionType, actionTarget, actionTask, resultClass,
                dialogController);
    }

    @Override
    protected void setActionHookList(ActionHookList<V> actionHookList) {
        super.setActionHookList(actionHookList);
        if (actionHookList != null) {
            obtainableHooks = actionHookList.getObtainableHooks();
        }
    }

    /**
     * Returns the list of the obtainable hooks.
     *
     * @return the obtainable hook or {@code null} if no such hook exists.
     */
    @Override
    List<Obtainable> getObtainableHooks() {
        return obtainableHooks;
    }

    /**
     * The obtaining state - performs the obtain action and invokes the hook.
     */
    protected Callable<Object[]> obtaining = new Callable<Object[]>() {
        @Override
        @SuppressWarnings("deprecation")
        public Object[] call() throws Exception {
            ObjectSelection objectSelection = null;
            ClientContext clientContext = null;

            if (!CANCELLED) {
                JVineController jVineController = JVineController
                        .getInstance(getController());
                G9Action<ObjectSelection> obtainAction;

                ActionTask<ObjectSelection> obtainActionTask = jVineController.getObtainDialogTask(getActionTarget(), getActionType());
                obtainAction = jVineController.getAction(ActionType.OBTAIN,
                        getActionTarget(), ObjectSelection.class,
                        obtainActionTask);

                objectSelection = obtainAction.call();
                objectSelection = obtained(objectSelection);

                clientContext = getController().getApplicationController()
                        .getClientContext();
                clientContext = contextHook(clientContext);

                log.info(ObtainableAction.this + " obtained");
            }

            Object[] returnValue = {
                    objectSelection, clientContext
            };
            return returnValue;
        }

    };

    @Override
    public V call() throws Exception {
        log.info("Starting execution of " + this);
        V result = null;

        try {
//            initializing.call();
            checking.call();
            Object obtainResult = obtaining.call();
            getActionTask().setTaskObject(obtainResult);
            try {
//                result = performing.call();
            } catch (CancelException ce) {
                log.info(this + " cancelled while performing.");
                if (log.isTraceEnabled()) {
                    log.trace("Cancel trace:", ce);
                }
                cancel();
            }
//            succeeding.call();
//            cancelling.call();
        } catch (Exception e) {
            failed(e);
        } finally {
//            finishing.call();
            cleanUpHook();

        }

        return result;
    }

    /**
     * The obtained hook. Invoked before the performed hook.
     *
     * @param objectSelection the obtained object selection
     * @return the object selection
     * @throws Exception re-throws exception from hook method.
     */
    ObjectSelection obtained(final ObjectSelection objectSelection)
            throws Exception {
        if (shouldInvokeHook() && !getObtainableHooks().isEmpty()) {
            String methodName = "obtained";
            Class[] params = {
                ObjectSelection.class
            };
            ThreadType threadType = getThreadType(methodName, params);
            HookMethod<ObjectSelection> obtainHook = new HookMethod<ObjectSelection>(
                    methodName) {
                @Override
                public ObjectSelection call() throws Exception {
                    ObjectSelection os = objectSelection;
                    for (Obtainable hook : getObtainableHooks()) {
                        os = hook.obtained(os);
                    }
                    return os;
                }
            };
            return hookInvoker.execute(getApplicationController(), threadType,
                    obtainHook);

        }

        return objectSelection;

    }

    /**
     * Invoke the client context hook
     *
     * @param clientContext the current client context
     * @return the client context to use
     * @throws Exception all exceptions are re-thrown.
     */
    ClientContext contextHook(final ClientContext clientContext)
            throws Exception {
        if (shouldInvokeHook() && !getObtainableHooks().isEmpty()) {
            String methodName = "contextHook";
            Class[] params = {
                ClientContext.class
            };
            ThreadType threadType = getThreadType(methodName, params);
            HookMethod<ClientContext> contextHook = new HookMethod<ClientContext>(
                    methodName) {
                @Override
                public ClientContext call() throws Exception {
                    ClientContext ctx = clientContext;
                    for (Obtainable hook : getObtainableHooks()) {
                        ctx = hook.contextHook(ctx);
                    }
                    return ctx;
                }
            };

            return hookInvoker.execute(getApplicationController(), threadType,
                    contextHook);
        }

        return clientContext;
    }

}