/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.util.HashMap;
import java.util.Map;

/**
 * PaginatorEvent is used to signal that a paginator navigation button has been clicked.
 *
 */
public final class PaginatorEvent {

    /** Enumerates the paging buttons */
    public enum NavigationButton {
        /** Go to next page */
        NEXT_PAGE,
        /** Skip forward */
        FAST_FORWARD,
        /** Skip to last page */
        LAST_PAGE,
        /** Go to previous page */
        PREVOIOUS_PAGE,
        /** Skip backwards */
        FAST_BACKWARDS,
        /** Skip to first page */
        FIRST_PAGE;
    }

    private static Map<String, NavigationButton> descrToButton = new HashMap<String, NavigationButton>();

    // Initiate the map from descriptive text to NavitagionButton enumerator.
    static {
        descrToButton.put("next", NavigationButton.NEXT_PAGE);
        descrToButton.put("fastf", NavigationButton.FAST_FORWARD);
        descrToButton.put("last", NavigationButton.LAST_PAGE);
        descrToButton.put("previous", NavigationButton.PREVOIOUS_PAGE);
        descrToButton.put("fastr", NavigationButton.FAST_BACKWARDS);
        descrToButton.put("first", NavigationButton.FIRST_PAGE);

    }

    /**
     * Get a navigation button based on a descriptive text.
     *
     * @param buttonDescription text describing the navigation button
     * @return the navigation button matching the description
     */
    public static NavigationButton getNavigationButton(String buttonDescription) {
        return descrToButton.get(buttonDescription);
    }

    private final int currentPage;

    private final NavigationButton navigationButton;

    private final Object sourceEvent;

    private final int pageCount;

    private final int maxPages;

    /**
     * Create a new paginator event.
     *
     * @param navigationButton the selected navigation button
     * @param currentPage the current paginator page
     * @param pageCount number of displayed pages
     * @param maxPages the maximum number of pages to be displayed in the paginator
     * @param sourceEvent the original paginator event
     */
    public PaginatorEvent(NavigationButton navigationButton, int currentPage, int pageCount, int maxPages, Object sourceEvent) {
        this.navigationButton = navigationButton;
        this.currentPage = currentPage;
        this.pageCount = pageCount;
        this.maxPages = maxPages;
        this.sourceEvent = sourceEvent;
    }

    /**
     * Construct a new paginator event.
     *
     * @param navigationButton the selected navigation button
     * @param currentPage the current paginator page
     * @param pageCount number of displayed pages
     * @param maxPages the maximum number of pages to be displayed in the paginator
     * @param sourceEvent the original paginator event
     */
    public PaginatorEvent(String navigationButton, int currentPage, int pageCount, int maxPages, Object sourceEvent) {
        this(getNavigationButton(navigationButton), currentPage, pageCount, maxPages, sourceEvent);
    }

    /**
     * Get the (target specific) source event.
     *
     * @return the source event
     */
    public Object getSourceEvent() {
        return sourceEvent;
    }

    /**
     * Get the current paginator page.
     *
     * @return the current page
     */
    public int getCurrentPage() {
        return currentPage;
    }

    /**
     * Get the number of displayed pages.
     *
     * @return the page count
     */
    public int getPageCount() {
        return pageCount;
    }

    /**
     * Get the maximum number of displayed pages.
     *
     * @return the max pages
     */
    public int getMaxPages() {
        return maxPages;
    }

    /**
     * Get the navigation button that originated this event.
     *
     * @return the selected navigation button
     */
    public NavigationButton getNavigationButton() {
        return navigationButton;
    }

    @Override
    public String toString() {
        return "PaginatorEvent [currentPage=" + currentPage + ", navigationButton=" + navigationButton + ", sourceEvent="
                + sourceEvent + "]";
    }

}
