/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import no.g9.support.convert.ConvertException;
import no.g9.support.convert.Converter;

/**
 * Represents a parameter binding in a remote service invocation. A parameter
 * binding consists of a formal parameter, an actual parameter and possibly a
 * converter.
 * <ol>
 * <li>The actual parameter is the model type of the parameter binding, e.g. an
 * object selection role, a role-attribute or a constant.
 * <li>The formal parameter is the declared interface-method attribute.
 * <li>A converter might be used to covert between actual and formal parameter.
 * </ol>
 * 
 * @param <A>
 *            the actual parameter type.
 */
public interface ParameterBinding<A> {

    /**
     * Boolean constant denoting that a parameter binding is in fact a return
     * value.
     */
    static final boolean RETURN_VALUE = true;

    /**
     * Test if this remote service parameter has a registered converter.
     * 
     * @return <code>true</code> if the remote service parameter has a converter
     */
    boolean hasConverter();

    /**
     * Return the converter to use with this remote service parameter. The
     * converter is used to convert between actual parameter (the converter's
     * model type) type and formal parameter type (the converter's target type).
     * 
     * @return the converter or <code>null</code> if no such converter exists.
     */
    Converter<A, ?> getConverter();

    /**
     * Get the actual parameter of this parameter binding.
     * 
     * @return the actual parameter.
     */
    Parameter<A> getActualParameter();

    /**
     * Get the declared name of the formal parameter
     * 
     * @return the formal parameter name.
     */
    String getFormalParameterName();

    /**
     * Test if this parameter binding represents the return value.
     * 
     * @return <code>true</code> if this binding is the return value.
     */
    boolean isReturnValue();

    /**
     * Get the formal parameter value of this parameter binding. If this
     * parameter binding has a converter, the actual parameter value will be
     * converted and returned. Otherwise the actual parameter value will be
     * casted to the formal parameter type and returned.
     * <p>
     * Note that unless the the parameter has been obtained, or the actual
     * parameter is a constant, the parameter returned value will most likely be
     * <code>null</code> (depending on the converter).
     * 
     * @param <F>
     *            the formal parameter type.
     * @return the converted parameter value.
     * @throws ConvertException
     *             exceptions thrown by converter are re-thrown.
     */
    <F> F getFormalParameterValue() throws ConvertException;

    /**
     * Obtain the actual parameter value.
     * 
     * @return the obtained value
     * 
     */

    /**
     * Get the actual parameter value of this parameter binding. If the
     * parameter binding has a converter, the formal parameter value will be
     * converted and returned. Otherwise the formal parameter value will be
     * casted to the actual parameter type and returned.
     * 
     * @param <F>
     *            the formal parameter type
     * @param formalParameterValue
     *            the formal parameter value
     * @return the actual parameter value
     * @throws ConvertException
     *             exceptions thrown by converter are re-thrown.
     */
    <F> A getActualParameterValue(F formalParameterValue)
            throws ConvertException;

    /**
     * Set the actual parameter value.
     * 
     * @param actualParameterValue
     *            the value of the actual parameter.
     */
    void setActualParameterValue(A actualParameterValue);

    /**
     * Set the formal parameter value. If the parameter binding has a converter,
     * the formal parameter value will be converted to the actual parameter
     * type.
     * 
     * @param <F>
     *            the formal parameter type.
     * @param formalParameterValue
     *            the formal parameter value.
     * @throws ConvertException
     *             exceptions thrown by converter are re-thrown.
     */
    <F> void setFormalParameterValue(F formalParameterValue)
            throws ConvertException;

    /**
     * Obtain the actual parameter value.
     * 
     * @return the obtained value
     * 
     */
    A obtainParameterValue();

}
