/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.communication;

import no.g9.support.ActionType;

/**
 * Payload from a SystemMessage as used in actions.
 */
public class G9ActionPayload {

    private enum supportedActionType {

        Open(ActionType.OPEN),
        Close(ActionType.CLOSE),
        Show(ActionType.SHOW),
        Hide(ActionType.HIDE),
        Activate(ActionType.ACTIVATE),
        Not_supported(null);

        public final ActionType actionType;

        private supportedActionType(ActionType actionType) {
            this.actionType = actionType;
        }

        public static supportedActionType valueOf(ActionType actionType) {
            switch (actionType) {
            case OPEN:
                return Open;
            case CLOSE:
                return Close;
            case SHOW:
                return Show;
            case HIDE:
                return Hide;
            case ACTIVATE:
                return Activate;
            default:
                return Not_supported;
            }
        }
    }

    /**
     * The action type.
     */
    public final ActionType actionType;

    /**
     * The payload.
     */
    public final String payload;

    /**
     * The valid flag.
     */
    public final boolean isValid;

    /**
     * Creates a g9 action payload from its string representation.
     *
     * @param payload The string representation of this action
     */
    public G9ActionPayload(String payload) {
        String[] split = payload.split("[\\/\\\\]", 2);
        String action = (split.length >= 1) ? split[0] : "";
        supportedActionType supType = supportedActionType.valueOf(action);

        if (supType != supportedActionType.Not_supported) {
            actionType = supType.actionType;
            isValid = true;
        } else {
            actionType = null;
            isValid = false;
        }
        this.payload = (split.length >= 2) ? split[1] : "";
    }

    /**
     * Creates a g9 action payload from a g9 action type.
     *
     * @param actionType The actiontype
     * @param containedPayload The payload of this action
     */
    public G9ActionPayload(ActionType actionType, String containedPayload) {
        this.actionType = actionType;
        this.payload = containedPayload;
        this.isValid = (supportedActionType.Not_supported != supportedActionType.valueOf(actionType));
    }

    @Override
    public String toString() {
        return code();
    }

    /**
     * @return The coded verion of this object
     */
    public String code() {
        supportedActionType supActionType = supportedActionType.valueOf(this.actionType);
        return supActionType + "/" + payload;
    }

    /**
     * Creates a new G9ActionPayload based on this with 
     * a new payload.
     * @param payload the new payload.
     * @return the new G9ActionPayload.
     */
    public G9ActionPayload payload(String payload) {
        return new G9ActionPayload(actionType,payload);
    }
}
