/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.communication;

/**
 * Used to code/decode messages. This class is immutable
 *
 */
public final class SystemMessage {

    /**
     * The system will forward the message to any system implementing this port.
     */
    public static final String ANY_RECEIVER = "";

    /**
     * The system address
     */
    public static final String SYSTEM = "system";

    /**
     * Send messages to this address to set the anchor part of the url to a message.
     */
    public static final String URL_RECEIVER = SYSTEM;

    /**
     * Send messages to this address to set the anchor part of the url to a message.
     */
    public static final String URL_PORT = "url";

    /**
     * Send messages to this address to set the anchor part of the url to a message.
     */
    public static final SystemMessage URL_TEMPLATE = new SystemMessage(URL_RECEIVER+"/"+URL_PORT);

    /**
     * Send messages to this address to push or pop the application stack.
     */
    public static final String STACK_RECEIVER = SYSTEM;    

    /**
     * Send messages to this address to push the application stack.
     */
    public static final String STACK_PUSH_PORT = "stack_push";

    /**
     * Send messages to this address to elevate the application stack.
     */
    public static final String STACK_ELEVATE_PORT = "stack_elevate";

    /**
     * Send messages to this address to pop the application stack.
     */
    public static final String STACK_POP_PORT = "stack_pop";

    /**
     * Send messages to this address to push the application stack.
     */
    public static final SystemMessage STACK_PUSH_TEMPLATE = new SystemMessage(STACK_RECEIVER + "/" + STACK_PUSH_PORT);

    /**
     * Send messages to this address to elevate the application stack.
     */
    public static final SystemMessage STACK_ELEVATE_TEMPLATE = new SystemMessage(STACK_RECEIVER + "/" + STACK_ELEVATE_PORT);

    /**
     * Send messages to this address to pop the application stack.
     */
    public static final SystemMessage STACK_POP_TEMPLATE = new SystemMessage(STACK_RECEIVER + "/" + STACK_POP_PORT);

    /**
     * Send messages to this address to query the URL of the system setup.
     */
    public static final String QUERY_SETUP_RECEIVER = SYSTEM;

    /**
     * Send messages to this address to query the URL of the system setup.
     */
    public static final String QUERY_SETUP_PORT = "system";

    /**
     * Send messages to this address to query the URL of the system setup.
     */
    public static final SystemMessage QUERY_SETUP_TEMPLATE = new SystemMessage(QUERY_SETUP_RECEIVER+"/"+QUERY_SETUP_PORT);

    /**
     * Menu items use this port to execute the event of a message.
     */
    public static final String MENU_PORT = "_application_menu";

    /**
     * This port is used to handle call backs in dialog interactions.
     */
    public static final String CALLBACK_PORT = "_call_back";

    /**
     * Send messages to this address for call back in dialog interactions.
     * Note: the application name must be added as a receiver.
     */
    public static final SystemMessage CALLBACK_TEMPLATE = new SystemMessage("/"+CALLBACK_PORT);

    /**
     * This port is used to handle errors from the system.
     */
    public static final String ERROR_PORT = "_error_handler";
    
    /**
     * This port is used for displaying alert messages in the UA.
     */
    public static final String ALERT_PORT = "alert";
    
    /**
     * Send messages to this address for the alert port.
     */
    public static final String ALERT_RECEIVER = SYSTEM;    

    /**
     * Send messages to this address for displaying alert messages in the UA.
     */
    public static final SystemMessage ALERT_TEMPLATE = new SystemMessage(ALERT_RECEIVER+"/"+ALERT_PORT);

    /**
     * This port is used for opening a URL in a new tab or window.
     */
    public static final String OPEN_PORT = "open";
    
    /**
     * Send messages to this address for the open port.
     */
    public static final String OPEN_RECEIVER = SYSTEM;    

    /**
     * Send messages to this address opening a URL in a new tab or window.
     */
    public static final SystemMessage OPEN_TEMPLATE = new SystemMessage(OPEN_RECEIVER+"/"+OPEN_PORT);

    /**
     * This port is used for setting the active dialog instance (for Close, Hide and Show actions).
     */
    public static final String ACTIVE_INSTANCE_PORT = "_active_instance";
    
    /**
     * Send messages to this address for setting the active dialog instance.
     */
    public static final SystemMessage ACTIVE_INSTANCE_TEMPLATE = new SystemMessage("/" + ACTIVE_INSTANCE_PORT);
    
    /**
     * The event method which performs actions received in messages.
     */
    public static final String MESSAGE_ACTION_PERFORMER = "_perform_message_action";

    /**
	 * The receiver
	 */
	public final String receiver;

	/**
	 * The port
	 */

	public final String port;
	/**
	 * The payload
	 */
	public final String payload;

	/**
	 * Constructor creating 
	 * @param codedMessage The coded message
	 */
	public SystemMessage(String codedMessage) {
		String[] split=codedMessage.split("[\\/\\\\]", 3);
		receiver=(split.length>=1)?split[0]:""; 
        port=(split.length>=2)?split[1]:"";
        payload=(split.length>=3)?split[2]:"";
	}

	/**
	 * @param receiver the receiver
	 * @param port the port
	 * @param payload the payload
	 */
	public SystemMessage(String receiver, String port, String payload) {
        this.receiver=receiver;
        this.port=port;
        this.payload=payload;
    }

    /**
	 * Codes this message
	 * @return The coded message
	 */
	public String code() {
		return receiver+"/"+port+"/"+payload;
	}

	@Override
    public String toString() {
	    return "SystemMessage: " + code();
	}

	/**
	 * Creates a new message based on this but changes the receiver
	 * @param receiver the new receiver
	 * @return the new message
	 */
	public SystemMessage receiver(String receiver){
        return new SystemMessage(receiver,port,payload);
	}

	/**
	 * Creates a new message based on this but changes the port
	 * @param port the new port
	 * @return the new message
	 */
	public SystemMessage port(String port){
	    return new SystemMessage(receiver,port,payload);
	}

	/**
	 * Creates a new message based on this but changes the payload
	 * @param payload the new payload
	 * @return the new message
	 */
	public SystemMessage payload(String payload){
	    return new SystemMessage(receiver,port,payload);
	}

	/**
	 * Creates a new message based on this but changes the payload
	 * @param payload the new payload
	 * @return the new message
	 */
	public SystemMessage payload(SystemMessage payload){
	    return new SystemMessage(receiver,port,payload.code());
	}
	
}
