/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.controller;

import no.g9.message.Message;

/**
 * Facilitates communication between two dialogs. This class wraps the dialog
 * controller, the call back value
 */
/**
 * @author ab
 *
 */
public class DialogCallBackValue {


    /** The source */
    private DialogController source;

    /** The value */
    private Object value;

    /* Optional parameters */

    /** Optional code */
    private Integer opCode;

    /** The message */
    private Message message;

    /** The flag value */
    private Flag flagValue = Flag.DEFAULT;

    /**
     * Enumerates possible flag values which can be used to signal the outcome
     * of the invoked dialog.
     */
    public enum Flag {
        /** Outcome success */
        SUCCESS,

        /** Outcome fail */
        FAILED,

        /** Outcome default */
        DEFAULT,
        ;
    }

    /**
     * Creates a new call back value with the specified source and value, and a
     * <code>null</code> opCode value.
     *
     * @param source
     *            the source of the call back value, e.g. the dialog
     *            <em>doing</em> the call back.
     * @param value
     *            the call back value.
     */
    public DialogCallBackValue(DialogController source, Object value) {
        this(source, value, null);
    }

    /**
     * Creates a new call back value with the specified source, value and opCode
     *
     * @param source
     *            the source of the call back value, e.g. the dialog
     *            <em>doing</em> the call back.
     * @param value
     *            the call back value.
     * @param opCode
     *            the opCode
     */
    public DialogCallBackValue(DialogController source, Object value,
            Integer opCode) {
        this(source, value, opCode, null);
    }

    /**
     * Creates a new call back value with the specified source, value and opCode
     *
     * @param source
     *            the source of the call back value, e.g. the dialog
     *            <em>doing</em> the call back.
     * @param value
     *            the call back value.
     * @param opCode
     *            the opCode
     * @param message
     *            the message
     */
    public DialogCallBackValue(DialogController source, Object value,
            Integer opCode, Message message) {
        this(source, value, opCode, message, null);
    }

    /**
     * Creates a new call back value with the specified source, value and opCode
     *
     * @param source
     *            the source of the call back value, e.g. the dialog
     *            <em>doing</em> the call back.
     * @param value
     *            the call back value.
     * @param opCode
     *            the opCode
     * @param message
     *            the message
     * @param flagValue the flag value
     */
    public DialogCallBackValue(DialogController source, Object value,
            Integer opCode, Message message, DialogCallBackValue.Flag flagValue) {
        this.source = source;
        this.value = value;
        this.opCode = opCode;
        this.message = message;
        if (flagValue != null) {
            this.flagValue = flagValue;
        }
    }

    /**
     * @return the opCode
     */
    public Integer getOpCode() {
        return opCode;
    }

    /**
     * @param <T>
     *            The dialog controller
     * @return the source of the call back value, e.g. the dialog <em>doing</em>
     *         the call back.
     * @throws ClassCastException
     *             if returned dialog controller does not match the expected
     *             dialog controller.
     */
    @SuppressWarnings("unchecked")
    public <T extends DialogController> T getSource() {
        return (T) source;
    }

    /**
     * Returns the call back value. The call back value is set when creating a
     * <em>CallBackValue</em> instance.
     *
     * @param <V>
     *            The expected value type.
     * @return the return value.
     *
     * @throws ClassCastException
     *             if the returned type does not match the expected type.
     */
    @SuppressWarnings("unchecked")
    public <V> V getValue() {
        return (V) value;
    }

    /**
     * Get the dialog call back value message.
     *
     * @return the message
     */
    public Message getMessage() {
        return message;
    }

    /**
     * Get the flag value used to indicate the outcome of this call back.
     * @return the flag value
     */
    public Flag getFlagValue() {
        return flagValue;
    }

    /**
     * Set the flag value.
     * @param flagValue the flag value
     */
    public void setFlagValue(Flag flagValue) {
        this.flagValue = flagValue;
    }

    /**
     * Set the op code of this call back.
     * @param opCode the op code
     */
    public void setOpCode(Integer opCode) {
        this.opCode = opCode;
    }

    /**
     * @param source the source to set
     */
    public void setSource(DialogController source) {
        this.source = source;
    }

    /**
     * Set the message of this call back.
     * @param message the message
     */
    public void setMessage(Message message) {
        this.message = message;
    }

    @Override
    public String toString() {
        return "DialogCallBackValue [source=" + source + ", value=" + value
                + ", opCode=" + opCode + ", message=" + message
                + ", flagValue=" + flagValue + "]";
    }

}
