/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.converter;

import java.util.Date;

import no.esito.log.Logger;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertContext;
import no.g9.support.convert.ConvertException;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;

/**
 * Converts between a Java 8 LocalTime attribute in the domain model and a Date
 * in the view (target). This converter is used for LocalTime attributes in the
 * DateTimeEntry component, as it does not support Java 8 date types.
 *
 * If there already is another converter present on the attribute, it is set as
 * a chained converter.
 */
public class LocalTimeConverter implements AttributeConverter<LocalTime, Date> {

	private AttributeConverter<LocalTime, Date> chainedConverter;

	private static final Logger log= Logger.getLogger(LocalTimeConverter.class);

	private static final int A_YEAR= 2000;
	private static final int A_MONTH= 1;
	private static final int A_DAY= 1;

	/**
	 * Create a new converter without a chained converter.
	 */
	public LocalTimeConverter() {
	}

	/**
	 * Create a new converter with the given chained converter. The model type
	 * is set to the model type of the chained converter. If the chained
	 * converter is null, the model type is set from the second parameter.
	 *
	 * @param chainedConverter
	 *            the chained converter, or null
	 */
	public LocalTimeConverter(
			AttributeConverter<LocalTime, Date> chainedConverter) {
		super();
		if (chainedConverter != null) {
			this.chainedConverter= chainedConverter;
		}
	}

	@Override
	public Date fromModel(LocalTime value, ConvertContext context)
			throws ConvertException {
		if (log.isTraceEnabled()) {
			log.trace("Convert from model: " + value);
		}
		if (chainedConverter != null) {
			Date fromModel= chainedConverter.fromModel(value, context);
			if (fromModel == null) {
				return null;
			}
			return fromModel;
		}
		if (value == null) {
			return null;
		}
		Instant i= value.atDate(LocalDate.of(A_YEAR, A_MONTH, A_DAY))
				.atZone(ZoneId.systemDefault()).toInstant();
		return new Date(i.toEpochMilli());
	}

	@Override
	public LocalTime toModel(Date value, ConvertContext context)
			throws ConvertException {
		if (log.isTraceEnabled()) {
			log.trace("Convert to model: " + value);
		}
		if (chainedConverter != null) {
			return chainedConverter.toModel(value, context);
		}
		if (value == null)
			return null;
		Instant i= Instant.ofEpochMilli(value.getTime());
		return LocalDateTime.ofInstant(i, ZoneId.systemDefault()).toLocalTime();
	}

	@Override
	public Class<Date> getTargetType() {
		return Date.class;
	}

	@Override
	public Class<LocalTime> getModelType() {
		return LocalTime.class;
	}

}
