/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.util;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import no.esito.util.DateUtil;
import no.g9.client.core.controller.DialogController;

/**
 * The Class QueryBuilder is used for constructing Query for AutoCompleteComponent.
 *
 *
 * buildAutoCompleteQuery - this method constructs the query for the given
 * 							columnAttribute, valueAttribute
 */
public class QueryBuilder {

	/**
	 * Builds the auto complete query.
	 *
	 * @param valueAttribute the value attribute
	 * @param columnAttributes the column attributes
	 * @param columnValue the column value
	 * @param dialogController the dialog controller
	 * @return the Object[] -- Object[0] is the query
	 * 						-- Object[1] is the parameters
	 */
	public static Object[] buildAutoCompleteQueryWithParameters(String valueAttribute, List<String> columnAttributes, String columnValue,
			DialogController dialogController) {
		RoleNameHelper roleNameHelper = new RoleNameHelper(dialogController);

		// Table name to construct the Query
		String tableName = roleNameHelper.getTableName(valueAttribute);
		// Column Names to Construct Query
		List<Object[]> colNames = getColumnNamesAndDataTypes(valueAttribute, columnAttributes, roleNameHelper);

		return buildQueryWithParameters(columnValue, tableName, colNames);
	}

	/**
	 * Builds the auto complete query.
	 *
	 * @param valueAttribute the value attribute
	 * @param columnAttributes the column attributes
	 * @param columnValue the column value
	 * @param dialogController the dialog controller
	 * @return the string
	 */
	public static String buildAutoCompleteQuery(String valueAttribute, List<String> columnAttributes, String columnValue,
			DialogController dialogController) {
		RoleNameHelper roleNameHelper = new RoleNameHelper(dialogController);
		// Table name to construct the Query
		String tableName = roleNameHelper.getTableName(valueAttribute);
		// Column Names to Construct Query
		List<Object[]> colNames = getColumnNamesAndDataTypes(valueAttribute, columnAttributes, roleNameHelper);

		return buildQuery(columnValue, tableName, colNames);
	}

	/**
	 * Builds the query.
	 *
	 * @param columnValue the column value
	 * @param tableName the table name
	 * @param colNames the column names and DataTypes
	 * @return  query string
	 */
	private static String buildQuery(String columnValue, String tableName,
			List<Object[]> colNames) {
		StringBuilder queryString = new StringBuilder("select ");
		boolean firstColumn = true;
		StringBuilder conditionString = new StringBuilder("");
		String orderBy="";
		for (Object[] columnName : colNames) {
			if (firstColumn) {
				firstColumn = false;
				orderBy=" order by alias." + columnName[0]+ " asc ";
			} else {
				queryString.append(", ");
				conditionString.append("OR ");
			}
			queryString.append("alias." + columnName[0]);
			// IF search is done on a Integer field then the field value has to be converted to String
			// because like operator is not applicable on Integer fields
			if(!(columnName[1].equals("java.lang.String"))){
			conditionString.append("(lower(str(alias." + columnName[0]
					+ ")) like '" + columnValue.toLowerCase()+"%" + "') ");
			}else{
				conditionString.append("(lower(alias." + columnName[0]
				    + ") like '" + columnValue.toLowerCase()+"%" + "') ");
			}
		}
		queryString.append(" from " + tableName + " as alias where ");
		queryString.append(conditionString);
		queryString.append(orderBy);
		return queryString.toString();
	}

	private static Object[] buildQueryWithParameters(String columnValue, String tableName,
			List<Object[]> colNames) {
		StringBuilder queryString = new StringBuilder("select ");
		boolean firstColumn = true;
		StringBuilder conditionString = new StringBuilder("");
		int parameterPosition = 1;
		String orderBy="";
		List<Object> valueList = new ArrayList<Object>();
		Object[] objArr = new Object[2];
		for (Object[] columnName : colNames) {
			if (firstColumn) {
				firstColumn = false;
				orderBy=" order by alias." + columnName[0]+ " asc ";
			} else {
				queryString.append(", ");
				if ((columnName[1].equals("java.util.Date"))) {
					Date date = DateUtil.stringToDate(columnValue,
							columnName[2].toString());
					if (date != null) {
						conditionString.append("OR ");
					}
				} else {
					conditionString.append("OR ");
				}
			}
			queryString.append("alias." + columnName[0]);

			// IF search is done on a Integer field then the field value has to be converted to String
			// because like operator is not applicable on Integer fields
			if((columnName[1].equals("java.lang.String"))){
				conditionString.append("(lower(alias." + columnName[0]
				    + ") like ?" + String.valueOf(parameterPosition++) + " ) ");
			valueList.add(columnValue.toLowerCase()+"%");
			} else if ((columnName[1].equals("java.util.Date"))) {
				Date date =DateUtil.stringToDate(columnValue,columnName[2].toString());
				if(date!=null){
				conditionString.append("(alias." + columnName[0]
						+ " between ?" + String.valueOf(parameterPosition++) + " and ?" + String.valueOf(parameterPosition++) + " ) ");
				valueList.add(DateUtil.startOfDay(date));
				valueList.add(DateUtil.endOfDay(date));
				}
			}
			else if (columnName[1].equals("java.lang.Integer")
					|| columnName[1].equals("java.lang.Long")
					|| columnName[1].equals("java.lang.Short")
					|| columnName[1].equals("no.g9.support.Numeric")) {
				conditionString.append("(lower(trim(str(alias." + columnName[0] + "))) like ?" + String.valueOf(parameterPosition++) + " ) ");
				valueList.add(columnValue.toLowerCase() + "%");
			}
			else {
				conditionString.append("(lower(str(alias." + columnName[0]
						+ ")) like ?" + String.valueOf(parameterPosition++) + " ) ");
				valueList.add(columnValue.toLowerCase() + "%");
			}
		}
		queryString.append(" from " + tableName + " as alias where ");
		queryString.append(conditionString);
		queryString.append(orderBy);
		objArr[0]=queryString.toString();
		objArr[1]=valueList;
		return objArr;
	}

	/**
	 * Gets the column names.
	 *
	 * @param valueAttribute the value attribute
	 * @param columnAttributes the column attributes
	 * @param roleNameHelper the role name helper
	 * @return the column names and Data types
	 */
	public static List<Object[]> getColumnNamesAndDataTypes(String valueAttribute, List<String> columnAttributes, RoleNameHelper roleNameHelper) {
		List<String> columnNames = new ArrayList<String>();
		columnNames.add(valueAttribute);
		if(columnAttributes.size() != 0){
			columnNames.addAll(columnAttributes);
		}
		else {
		    columnNames.add(valueAttribute);
		}
		Object[] colNames = null;
		List<Object[]> list = new ArrayList<Object[]>();
		for (int i = 0; i < columnNames.size(); i++) {
			colNames = roleNameHelper.getColumnName(valueAttribute, columnNames.get(i).trim());
			list.add(colNames);
		}
		return list;
	}

}
