/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import no.g9.support.convert.ConvertException;

/**
 * A field validator that returns true unless there is a converter specified for
 * the field, and a call to convert the candidate value (to model) results in a
 * ConvertException. The ConvertException for a failed attempt will be passed as
 * a message argument for the validation result.
 *
 * In other words, this validator returns true if the candidate value can be
 * converted to model representation.
 *
 * Example of use:
 * {@code
 * <bean id="convertValidator"
 *     class="no.g9.client.core.validator.ConvertValidator">
 * </bean>
 *
 * This bean supports the following properties:
 * <li> failMessageNumber: Message number to return on failed validations.}
 *
 */
public class ConverterValidator implements FieldValidator {

    private String failMessageNumber = ValidationResult.DEFAULT_FAIL_MESSAGE;

    /**
     * Sets message number to return on failed validations.
     * @param failMessageID The message number to return on failed validations
     */
    public void setFailMessageNumber(String failMessageID) {
        this.failMessageNumber = failMessageID;
    }

    @Override
    public ValidationResult validate(Object value, ValidateContext context) {
        try {
            context.convertToModel(value);
            return ValidationResult.DEFAULT_OK_RESULT;
        } catch (ConvertException e) {
            return new ValidationResult(false, failMessageNumber, new Object[] {e});
        }
    }

}
