/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.converter.FieldConvertContext;
import no.g9.client.core.message.FieldMessageContext;
import no.g9.client.core.view.ListRow;
import no.g9.os.AttributeConstant;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertException;

/**
 * Contains information about a field validation context
 */
public class ValidateContext implements FieldMessageContext {

	private final DialogController dialogController;
	private final DialogObjectConstant dialogObjectConstant;
	private final ListRow listRow;
	private final ActionType action;


	/**
	 * Constructs a new ValidationContext
	 *
	 * @param dCtrl
	 *            The DialogController of the context
	 * @param dObjectConstant
	 *            The DialogObjectConstant of the context
	 */
	public ValidateContext(final DialogController dCtrl,
			final DialogObjectConstant dObjectConstant) {
		this(dCtrl, null, dObjectConstant, null);
	}

	/**
	 * Construct a new Validate Context.
	 * @param dCtrl context's dialog controller
	 * @param dialogObject dialog object being validated
	 * @param actionType action type that lead to this validation
	 */
	public ValidateContext(DialogController dCtrl, DialogObjectConstant dialogObject, ActionType actionType) {
		this(dCtrl, null, dialogObject, actionType);
	}

	/**
	 * Constructs a new ValidationContext
	 *
	 * @param dCtrl
	 *            The DialogController of the context
	 * @param lRow
	 *            The ListRow of the context
	 * @param dObjectConstant
	 *            The DialogObjectConstant of the context
	 */
	public ValidateContext(final DialogController dCtrl, final ListRow lRow,
			final DialogObjectConstant dObjectConstant) {
		this(dCtrl, lRow, dObjectConstant, null);
	}

	/**
	 * Construct a new Validation Context.
	 * @param dCtrl dialog controller of the context
	 * @param lRow context's list row
	 * @param dObjectConstant context's dialog object
	 * @param action context's action
	 */
	public ValidateContext(DialogController dCtrl, ListRow lRow,
			DialogObjectConstant dObjectConstant, ActionType action) {
		dialogController = dCtrl;
		listRow = lRow;
		dialogObjectConstant = dObjectConstant;
		this.action = action;
	}

	/**
	 * Converts the specified view value to the corresponding domain value using
	 * the attribute's converter.
	 * @param <M> model
	 * @param <T> target
	 *
	 * @param value
	 *            the view value to convert.
	 * @return the converted value.
	 * @throws ConvertException
	 *             On all problems
	 */
    @SuppressWarnings("unchecked")
    public <M,T> M convertToModel(T value) throws ConvertException {
		AttributeConverter<M,T> converter = dialogController.getViewModel().getConverter(dialogObjectConstant);
		if (converter == null) {
			// No converter for this attribute...
			return (M) value;
		}
		FieldConvertContext context = new FieldConvertContext(dialogController,
				dialogObjectConstant);
		return converter.toModel(value, context);

	}

	/**
	 * @return the dialogController of the context
	 */
	public DialogController getDialogController() {
		return dialogController;
	}

	/**
	 * @return the dialogObjectConstant of the context
	 */
	public DialogObjectConstant getDialogObjectConstant() {
		return dialogObjectConstant;
	}

	/**
	 * Get the role of the dialog object's attribute.
	 * @return role of dialog object's attribute
	 */
	public RoleConstant getRole() {
		RoleConstant role = null;
		if (dialogObjectConstant != null) {
			AttributeConstant attribute = dialogObjectConstant.getAttribute();
			if (attribute != null) {
				role = attribute.getAttributeRole();
			}
		}
		return role;
	}

	/**
	 * @return the listRow of the context
	 */
	public ListRow getListRow() {
		return listRow;
	}

	/**
	 * Get the action type (if any) that initiated this validation.
	 * @return action type of the context
	 */
    public ActionType getAction() {
        return action;
    }

	@Override
	public DialogObjectConstant getField() {
		return this.dialogObjectConstant;
	}


}
