/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import java.util.ArrayList;
import java.util.List;

/**
 * A validator that chains other validators.
 * Can be used to apply multiple validators to an attribute.
 * A candidate value will pass through validation iff it passes through all
 * validations in chain. Otherwise, the validation result of the first failed
 * validation will be returned.
 *
 * Example of usage:
 * {@code
 * <bean id="validator1" ... />
 * <bean id="validator2" ... />
 * <bean id="someValidator"
 *     class="no.g9.client.core.validator.ValidatorChainValidator">
 *     <property name="validators">
 *         <list>
 *             <ref bean="validator1" />
 *             <ref bean="validator2" />
 *         </list>
 *     </property>
 * </bean>}
 *
 */
public class ValidatorChainValidator implements FieldValidator {

    private List<FieldValidator> validators;

    /**
     * @param validators the validators to set
     */
    public void setValidators(List<FieldValidator> validators) {
        this.validators = validators;
    }

    /**
     * Adds a validator to the chain.
     * @param validator The validator to add
     */
    public void addValidator(FieldValidator validator) {
        if (validators == null) {
            validators = new ArrayList<FieldValidator>();
        }
        validators.add(validator);
    }

    @Override
    public ValidationResult validate(Object value, ValidateContext context) {
        if (validators == null) return ValidationResult.DEFAULT_OK_RESULT;
        for (FieldValidator v : validators) {
            ValidationResult res = v.validate(value, context);
            if (res.succeeded() == false) {
                return res;
            }
        }
        return ValidationResult.DEFAULT_OK_RESULT;
    }

}
