/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view;

import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogInstance;
import no.g9.support.ActionType;

/**
 * The controller as seen from the view.
 *
 */
public interface ApplicationView {

    /**
     * Perform the specified action.
     *
     * @param action the action to perform.
     * @param target the action's target.
     * @deprecated use {@link #performAction(ActionType, DialogConstant, Boolean)}
     *
     */
    @Deprecated
    void performAction(ActionType action, DialogConstant target);

    /**
     * Perform the specified action.
     * @param action the action to perform
     * @param target the action's target
     * @param actionFlag general purpose flag
     */
    void performAction(ActionType action, DialogConstant target, Boolean actionFlag);

    /**
     * Open the given resource in the view.
     *
     * @param resourceId the id of the resource to open.
     * @param purge if true, remove the resource after use
     */
    void openResource(String resourceId, boolean purge);

    /**
     * Check if the given dialog is open.
     *
     * @param target the dialog to check.
     * @return true if open, false otherwise.
     */
    boolean isOpen(DialogConstant target);

    /**
     * Check if the given dialog is shown.
     *
     * @param target the dialog to check.
     * @return true if shown, false otherwise.
     */
    boolean isShown(DialogConstant target);

    /**
     * Open the given dialog.
     *
     * @param target the dialog to open.
     */
    public void open(DialogConstant target);

    /**
     * Open the given dialog.
     * @param target the dialog to open
     * @param openNew if <code>true</code> a <em>new</em> instance of the dialog will be created.
     */
    public void open(DialogConstant target, boolean openNew);

    /**
     * Close the given dialog.
     *
     * @param target the dialog to close.
     */
    public void close(DialogConstant target);

    /**
     * Show the given dialog.
     *
     * @param target the dialog to show.
     */
    public void show(DialogConstant target);

    /**
     * Hide the given dialog.
     *
     * @param target the dialog to hide.
     */
    public void hide(DialogConstant target);

    /**
     * Clear the given dialog.
     *
     * @param target the dialog to clear.
     */
    public void clearObject(DialogConstant target);

    /**
     * Get the ViewModel for the given dialog. Repeated calls with the same
     * dialog as argument must return the same ViewModel object.
     *
     * @param dialog the which "owns" the ViewModel.
     * @return the ViewModel for the given dialog.
     */
    ViewModel getViewModel(DialogConstant dialog);

    /**
     * Get the DialogView for the given dialog. Repeated calls with the same
     * dialog as argument must return the same DialogView object.
     * @param <T> the DialogView subclass to get.
     *
     * @param dialog the dialog which "owns" the DialogView.
     * @return the DialogView for the given dialog.
     */
    <T extends DialogView> T getDialogView(DialogConstant dialog);

    /**
     * Get the title for the given dialog instance.
     *
     * @param instance the dialog instance in question
     * @return the title for the instance
     */
    String getDialogTitle(DialogInstance instance);

    /**
     * Set the title to use for the given dialog instance.
     * If the title is null, the default dialog title will be used.
     *
     * @param instance the dialog instance in question
     * @param title the new title to use for the instance
     */
    void setDialogTitle(DialogInstance instance, String title);

    /**
     * Get the referenced application controller.
     *
     * @return the application controller
     */
    ApplicationController getApplicationController();

}