/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view;

import java.util.Collection;
import java.util.List;

import no.esito.jvine.controller.OSNode;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.support.ActionType;

/**
 * The view as seen from the controller.
 *
 */
public interface DialogView {

    /**
     * Invoked after the view has been fully initialized.
     */
    void init();

    /**
     * Notify the view that the given roles are updated.
     * The view should ask the controller to get the actual data for
     * the updated roles, and then display it.
     *
     * @param nodes a collection of (generated) role enums.
     */
    void update(Collection<OSNode<?>> nodes);

    /**
     * Perform the specified action.
     *
     * @param action the action to perform.
     * @param target the action's target.
     */
    void performAction(ActionType action, DialogObjectConstant target);

    /**
     * Check if the given dialog object is editable.
     *
     * @param target the dialog object to check.
     * @return true if editable, false otherwise.
     */
    boolean isEditable(DialogObjectConstant target);

    /**
     * Check if the given dialog object is enabled.
     *
     * @param target the dialog object to check.
     * @return true if enabled, false otherwise.
     */
    boolean isEnabled(DialogObjectConstant target);

    /**
     * Check if the given dialog object is mandatory.
     *
     * @param target the dialog object to check.
     * @return true if mandatory, false otherwise.
     */
    boolean isMandatory(DialogObjectConstant target);

    /**
     * Check if the given dialog object is shown.
     *
     * @param target the dialog object to check.
     * @return true if shown, false otherwise.
     */
    boolean isShown(DialogObjectConstant target);

    /**
     * Set the modal property for a dialog.
     *
     * @param modal true if the dialog should be modal
     */
    void setModal(boolean modal);

    /**
     * Check if the given dialog is modal.
     *
     * @return true if modal, false otherwise.
     */
    boolean isModal();

    /**
     * Get the effect object for the given dialog object.
     *
     * @param target the dialog object to check.
     * @return the effect object, or null.
     */
    Object getEffect(DialogObjectConstant target);

    /**
     * Set the editable property for the given dialog object.
     *
     * @param target the dialog object to set.
     * @param value the new value for the property.
     */
    void setEditable(DialogObjectConstant target, boolean value);

    /**
     * Set the mandatory property for the given dialog object.
     *
     * @param target the dialog object to set.
     * @param value the new value for the property.
     */
    void setMandatory(DialogObjectConstant target, boolean value);

    /**
     * Get the mandatory property for the given dialog object.
     * @param target the dialog object with said property
     * @return the property value
     */
    boolean getMandatory(DialogObjectConstant target);

    /**
     * Set the effect object for the given dialog object.
     *
     * @param target the dialog object to set.
     * @param effect the effect object to set.
     */
    void setEffect(DialogObjectConstant target, Object effect);

    /**
     * Enable the given dialog object.
     *
     * @param target the dialog object to enable.
     */
    void enable(DialogObjectConstant target);

    /**
     * Disable the given dialog object.
     *
     * @param target the dialog object to disable.
     */
    public void disable(DialogObjectConstant target);

    /**
     * Show the given dialog object.
     *
     * @param target the dialog object to show.
     */
    void show(DialogObjectConstant target);

    /**
     * Hide the given dialog object.
     *
     * @param target the dialog object to hide.
     */
    void hide(DialogObjectConstant target);

    /**
     * Set the given property value for a widget.
     *
     * @param target the dialog object to set the property for
     * @param key the property key
     * @param value the property value to set
     */
    void setWidgetProperty(DialogObjectConstant target, String key, Object value);

    /**
     * Get a property value for a widget.
     *
     * @param target the dialog object to get the property for
     * @param key the property key
     * @return the property value
     */
    Object getWidgetProperty(DialogObjectConstant target, String key);

    /**
     * Clear the given dialog object.
     *
     * @param target the dialog object to clear.
     */
    void clearObject(DialogObjectConstant target);

    /**
     * Set the current Notebook page to show. To be used for Notebooks without
     * tabs (widget stacks).
     *
     * @param notebook the Notebook.
     * @param page the new current page for the Notebook.
     */
    void setCurrentNotebookPage(DialogObjectConstant notebook,
            DialogObjectConstant page);

    /**
     * Get the current page for the given Notebook.
     *
     * @param notebook the Notebook in question.
     * @return the current page for the Notebook.
     */
    DialogObjectConstant getCurrentNotebookPage(DialogObjectConstant notebook);

    /**
     * Set a property for the given dialog component.
     * The available properties and property values is target dependent.
     *
     * @param target the dialog object to set the property on.
     * @param key the property to set.
     * @param value the new value to set for the property.
     */
    void setProperty(DialogObjectConstant target, String key, Object value);

    /**
     * @return the view model for the dialog.
     */
    ViewModel getViewModel();

    /**
     * Get the message title property for the specified dialog object.
     *
     * @param dialogObject the dialog object
     * @return the title or <code>null</code> if no title exist
     */
    String getTitle(DialogObjectConstant dialogObject);

    /**
     * Set the title to use for this dialog instance.
     * If the title is null, the default dialog title will be used.
     *
     * @param title the new title to use for the instance
     */
    void setDialogTitle(String title);

    /**
     * Gets the 1-based dialog instance number.
     *
     * @return the dialog instance number
     */
    int getDialogInstanceNumber();

    /**
     * Returns the list of all dialog object constants for the dialog.
     *
     * @return all dialog object constants
     */
    List<DialogObjectConstant> getDialogObjectConstants();

    /**
     * Dispatch an event to the dialog controller.
     * Checks that the dialog can dispatch events before proceeding.
     * Note that this method should only be used from a GUI thread.
     *
     * @param listRow the Table or List row in which the event occurred, or null if not in a table/list
     * @param trigger the cause of the event
     * @param event the type of event
     * @param method the event method name
     */
    void dispatchEvent(ListRow listRow, DialogObjectConstant trigger, Object event, String method);

    /**
     * Get the current default button for the dialog. The default button can be
     * set as a property on a Window in the dialog model, and can be changed
     * at runtime.
     *
     * @return the current default button for the dialog
     */
    DialogObjectConstant getDefaultButton();

    /**
     * Set the new default button for the dialog.
     * The default button should have a Clicked event in the model.
     *
     * @param defaultButton the new default button for the dialog
     */
    void setDefaultButton(DialogObjectConstant defaultButton);

}