/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Observable;

import no.esito.jvine.view.DefaultPropertyManager;
import no.esito.log.Logger;
import no.g9.client.core.controller.DialogObjectConstant;

/**
 * Holds data and a selected flag for a ListBlock row.
 */
public abstract class ListRow extends Observable implements PropertyManager {

    private static final Logger log = Logger.getLogger(ListRow.class);

    private int dialogInstanceNumber;

    private Boolean selected;

    private Map<DialogObjectConstant, Object> values = new HashMap<DialogObjectConstant, Object>();

    private final PropertyManager propertyManager = new DefaultPropertyManager();

    /**
     * Create a new ListRow with the given selected state.
     * 
     * @param dialogInstanceNumber the instance number for the owner
     * @param selected the selected state for the row
     */
    public ListRow(int dialogInstanceNumber, boolean selected) {
        this.dialogInstanceNumber = dialogInstanceNumber;
        setSelected(Boolean.valueOf(selected));
    }

    /**
     * @return the instance number for the owner dialog.
     */
    public int getDialogInstanceNumber() {
        return dialogInstanceNumber;
    }

    /**
     * @return the selected state as a primitive value.
     */
    public boolean isRowSelected() {
        return getSelected() != null && getSelected().booleanValue();
    }

    /**
     * Get the selected state for this row.
     * 
     * @return the selected state.
     */
    public Boolean getSelected() {
        return selected;
    }

    /**
     * Set the selected state for this row.
     * 
     * @param selected
     *            the new selected state.
     */
    public void setSelected(final Boolean selected) {
        this.selected = selected;
    }

    /**
     * Get the value for the given field.
     * 
     * @param field
     *            the field to get.
     * @return the field value.
     */
    public Object getValue(final DialogObjectConstant field) {
        return getValues().get(field);
    }

    /**
     * Set the value for the given field. Observers are notified.
     * 
     * @param field
     *            the field to set.
     * @param value
     *            the new field value.
     */
    public void setValue(final DialogObjectConstant field, final Object value) {
        if (log.isTraceEnabled()) {
            log.trace("setValue: " + field + "[" + value + "]");
        }
        if (getFields().contains(field)) {
            getValues().put(field, value);
            setChanged();
            notifyObservers();
        }
    }

    /**
     * @return a collection of the fields in this ListRow.
     */
    public abstract Collection<DialogObjectConstant> getFields();

    private Map<DialogObjectConstant, Object> getValues() {
        return values;
    }

    @Override
    public final void clear(DialogObjectConstant dialogObject, Property<?> property) {
        columnCheck(dialogObject);
        propertyManager.clear(dialogObject, property);

    }

    @Override
    public final void clearAll() {
        propertyManager.clearAll();
    }

    @Override
    public final void clearAll(DialogObjectConstant dialogConstant) {
        columnCheck(dialogConstant);
        propertyManager.clearAll(dialogConstant);
    }

    @Override
    public final <T> T getProperty(DialogObjectConstant dialogObject, Property<T> property) {
        columnCheck(dialogObject);
        return propertyManager.getProperty(dialogObject, property);
    }

    @Override
    public final String listProperties() {
        return propertyManager.listProperties();
    }

    @Override
    public final String listProperties(DialogObjectConstant dialogObject) {
        columnCheck(dialogObject);
        return propertyManager.listProperties(dialogObject);
    }

    @Override
    public final <T> void setProperty(DialogObjectConstant dialogObject, Property<T> property, T value) {
        columnCheck(dialogObject);
        propertyManager.setProperty(dialogObject, property, value);
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + getValues();
    }

    @Override
    public boolean isPropertySet(DialogObjectConstant dialogObject, Property<?> property) {
        columnCheck(dialogObject);
        return propertyManager.isPropertySet(dialogObject, property);
    }

    private void columnCheck(DialogObjectConstant dialogObject) {
        if (!values.keySet().contains(dialogObject)) {
            throw new NoSuchElementException("Unknown column: " + dialogObject);
        }
    }

    /**
     * Creates a bean reference that is a concatenation of the
     * <code>beanName</code> and the <code>dialogIntanceNumber</code>.
     * 
     * @param beanName
     *            the bean name
     * @return the string the correct bean ref for this instance.
     */
    protected String createBeanRef(String beanName) {
        String retVal = "#{" + beanName;
        if (getDialogInstanceNumber() > 1) {
            retVal = retVal + getDialogInstanceNumber();
        }
        return retVal + "}";
    }
}