/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view;

import no.g9.client.core.controller.DialogObjectConstant;

/**
 * Manages properties for DialogObjectConstants. 
 */
public interface PropertyManager {

    /**
     * Set the specified property.
     * 
     * @param <T> the property value type
     * @param dialogObject the dialog object whose property to set.
     * @param property the property to set.
     * @param value the property value.
     */
    public abstract <T> void setProperty(DialogObjectConstant dialogObject,
            Property<T> property, T value);

    /**
     * Get the specified property value. If the property is not set, the
     * property's default value is returned (use
     * {@link #isPropertySet(DialogObjectConstant, Property)} to
     * test if a property is set.
     * 
     * @param <T> the property value type.
     * @param dialogObject the dialog object whose property to get.
     * @param property the property to get.
     * @return the property value.
     * @throws ClassCastException if the actual property type does not match the
     *             expected property value type.
     */
    public abstract <T> T getProperty(DialogObjectConstant dialogObject,
            Property<T> property);

    /**
     * Clears all properties managed by this property manager.
     */
    public abstract void clearAll();

    /**
     * Clear all properties belonging to the specified dialog object.
     * 
     * @param dialogConstant the dialog object whose properties to clear.
     */
    public abstract void clearAll(DialogObjectConstant dialogConstant);

    /**
     * Clear the specified property of the specified dialog object.
     * 
     * @param dialogObject the dialog object whose property to clear.
     * @param property the property to clear.
     */
    public abstract void clear(DialogObjectConstant dialogObject,
            Property<?> property);

    /**
     * List all managed properties for the specified dialog object. This is
     * mainly intended for logging and debugging purposes.
     * 
     * @param dialogObject the dialog object whose properties to get.
     * @return a String containing the properties of the dialog object.
     */
    public abstract String listProperties(DialogObjectConstant dialogObject);

    /**
     * List all properties managed by this property manager. This is mainly
     * intended for logging and debugging purposes.
     * 
     * @return a String containing all dialog objects and properties.
     */
    public abstract String listProperties();
    
    /**
     * Test if a property is set.
     * 
     * @param dialogObject the dialog object that possibly has the property set.
     * @param property the property
     * @return <code>true</code> if the property is set, otherwise
     *         <code>false</code>.
     */
    public abstract boolean isPropertySet(DialogObjectConstant dialogObject,
        Property<?> property); 

}