/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view;

import java.security.SecureRandom;

/**
 * Defines a resource that can displayed by
 */
public final class Resource {

    private final String id;
    private final String resourceName;
    private final String mimeType;
    private final byte[] resource;

    private static class Lazy {
        static SecureRandom random = new SecureRandom();
    }

    private static String createID() {
        long nextLong = Lazy.random.nextLong();
        if (nextLong == Long.MIN_VALUE) {
            nextLong = 0; // corner case
        } else {
            nextLong = Math.abs(nextLong);
        }
        return Long.toString(nextLong);
    }

    /**
     * Construct a new resource.
     * 
     * @param resourceName user friendly name of resource
     * @param mimeType resource mime type
     * @param resource the actual resource as a byte array
     */
    public Resource(String resourceName, String mimeType, byte[] resource) {
        this.id = createID();
        this.resourceName = resourceName;
        this.mimeType = mimeType;
        this.resource = resource.clone();
    }

    /**
     * Get the resource id.
     * 
     * @return resource id
     */
    public final String getId() {
        return id;
    }

    /**
     * Get the resource name.
     * 
     * @return resource name
     */
    public final String getResourceName() {
        return resourceName;
    }

    /**
     * Get the resource mime type.
     * 
     * @return resource mime type
     */
    public final String getMimeType() {
        return mimeType;
    }

    /**
     * Get the resource byte array.
     * 
     * @return resource byte array
     */
    public final byte[] getResource() {
        return resource.clone();
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((id == null) ? 0 : id.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }

        if (!(obj instanceof Resource)) {
            return false;
        }

        Resource other = (Resource) obj;
        if (id == null && other.id != null) {
            return false;
        } else if (id != null && !id.equals(other.id)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "Resource [id=" + id + ", resourceName=" + resourceName
                + ", mimeType=" + mimeType + "]";
    }

}
