/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view;

import java.util.Collection;
import java.util.List;

import no.esito.jvine.controller.FieldData;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.validator.FieldValidator;
import no.g9.client.core.view.table.TableModel;
import no.g9.client.core.view.tree.TreeModel;
import no.g9.client.core.view.tree.TreeNode;
import no.g9.os.AttributeConstant;
import no.g9.os.RoleConstant;
import no.g9.support.convert.AttributeConverter;

/**
 * The view model interface, representing the public interface to the view model of a <code>DialogView</code>.
 */
public interface ViewModel {

    /**
     * Gets the specified field value.<br>
     * <strong>Note:</strong> The field value is the value as seen from the model. If the field attribute has a converter, this
     * may differ from the value as seen in the view, as the converter's <code>toModel</code> will be invoked for the field
     * value before it is returned.
     *
     * @param field The (generated) enum denoting the role attribute
     * @return the specified field value.
     */
    Object getFieldValue(DialogObjectConstant field);

    /**
     * Displays the specified field value. <br>
     * <strong>Note:</strong> The field value is the value as seen from the model. If the field attribute has a converter, this
     * may differ from the value as seen in the view, as the converter's <code>fromModel</code> will be invoked for the field
     * value.
     *
     * @param field The (generated) enum denoting the role attribute
     * @param fieldValue The field value to display
     */
    void setFieldValue(DialogObjectConstant field, Object fieldValue);

    /**
     * Return the changed status for the given field. A field is considered to be changed if the user has changed the field
     * value since the last display involving the field.
     *
     * @param field the field to check.
     * @return the changed status for the given field.
     */
    boolean isChanged(DialogObjectConstant field);

    /**
     * Set the changed status for the given field. The changed status is automatically handled by the view itself, but may be
     * overridden with this method.
     *
     * @param field the field to set.
     * @param changed the new changed status for the given field (true/false).
     */
    void setChanged(DialogObjectConstant field, boolean changed);

    /**
     * Get a collection of all changed fields for the dialog, ie. a collection containing all fields which returns true from the
     * isChanged method.
     *
     * @return a collection of all changed fields.
     */
    Collection<DialogObjectConstant> getChangedFields();

    /**
     * Get a collection of changed attributes for the given role. This is a convenience method, returning the attributes for the
     * fields of the given role from the getChangedFields method.
     *
     * @param role the role which owns the attributes.
     * @return a collection of changed attributes for the given role.
     */
    Collection<AttributeConstant> getChangedAttributes(RoleConstant role);

    /**
     * Get all fields which display values for the given attribute.
     *
     * @param attribute the attribute behind the fields.
     * @return the fields for the given attribute.
     */
    Collection<DialogObjectConstant> getAttributeFields(AttributeConstant attribute);

    /**
     * Get the displayed list of ListRows for the given role.
     *
     * @param <T> the actual return type (subclass of {@code List<ListRow>})
     * @param role the role which "owns" the list
     * @return the list of ListRows
     */
    <T extends ListRow> List<T> getDisplayList(RoleConstant role);

    /**
     * Get the TableModel for the given role.
     *
     * @param <T> actual generated subclass of ListRow
     * @param role the role which "owns" the TableModel
     * @return the table model used to hold list rows for the given role
     */
    <T extends ListRow> TableModel<T> getTableModel(RoleConstant role);

    /**
     * Set the TableModel for the given role.
     *
     * @param <T> actual generated subclass of ListRow
     * @param role the role which "owns" the list
     * @param tableModel the table model to set
     */
    <T extends ListRow> void setTableModel(RoleConstant role, TableModel<T> tableModel);

    /**
     * Get the TreeModel for the given treeview.
     *
     * @param <T> generated node that implements TreeNode
     * @param <L> list row
     * @param attribute the treeview dialog object constant
     * @return the tree model that holds the tree nodes for a given treeview widget
     */
	<T extends TreeNode, L extends ListRow> TreeModel<T, L> getTreeModel(DialogObjectConstant attribute);

    /**
     * Set the TreeModel for the given treeview.
     *
     * @param attribute the treeview dialog object constant
     * @param treeModel the tree model for a given treeview
     */
	void setTreeModel(DialogObjectConstant attribute, TreeModel<? extends TreeNode, ? extends ListRow> treeModel);

    /**
     * Clear all fields of the view model.
     */
    void clearViewModel();

    /**
     * Get the converter for the given field.
     * @param <M> model
     * @param <T> target
     *
     * @param field the field which has a converter.
     * @return the converter.
     */
    <M,T> AttributeConverter<M, T> getConverter(DialogObjectConstant field);

    /**
     * Get the validator for the given field.
     *
     * @param field the field which has a validator.
     * @return the validator.
     */
    FieldValidator getValidator(DialogObjectConstant field);

    /**
     * Get the collection of fields for the given role.
     *
     * @param role the role which owns the field collection.
     * @return the field collection.
     */
    Collection<DialogObjectConstant> getRoleFields(RoleConstant role);

    /**
     * Get the dialog controller for this view model.
     *
     * @return the dialog controller.
     */
    public DialogController getDialogController();

    /**
     * Get row field values for a role.
     *
     * @param row The row number.
     * @param role The (generated) enum denoting the role.
     * @return field data for the given row.
     */
    public FieldData getRowFieldData(ListRow row, RoleConstant role);

    /**
     * Get all field values for a role.
     *
     * @param role The (generated) enum denoting the role
     * @return a collection of instance field data populated by the view.
     */
    Collection<FieldData> getAllFieldData(RoleConstant role);

    /**
     * Get the field values for the current (selected) fields of a role.
     *
     * @param role The (generated) enum denoting the role
     * @return an instance field data populated by the view.
     */
    FieldData getCurrentFieldData(RoleConstant role);

}
