/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.menu;

import java.util.ArrayList;
import java.util.List;

import no.g9.client.core.view.ImageInfo;

/**
 * A menu item from the dialog model.
 */
public class Menu extends MenuBase {

    /**
     * The type of the menu.
     */
    public static enum MENU_TYPE {
        /** The File menu */
        FILEMENU,
        /** The Edit menu */
        EDITMENU,
        /** The View menu */
        VIEWMENU,
        /** The Window menu */
        WINDOWMENU,
        /** The Help menu */
        HELPMENU,
        /** The User menu */
        USERMENU
    }

    private int dialogInstance;

    private MENU_TYPE type;

    private List<MenuBase> children = new ArrayList<MenuBase>();

    /**
     * Create a new menu item without an event.
     *
     * @param id the ID of the menu
     * @param dialogId the ID of the dialog which owns the menu
     * @param title the title of the menu or menu item
     * @param mnemonic the mnemonic for the menu or menu item
     * @param accelerator the accelerator for the menu or menu item
     * @param style the style from the dialog model
     * @param menuType the type of the menu
     */
    @Deprecated
    public Menu(String id, String dialogId, String title, String mnemonic, String accelerator, String style, MENU_TYPE menuType) {
        this(id, dialogId, title, mnemonic, accelerator, style, null, null, menuType);
    }

    /**
     * Create a new menu item without an event.
     *
     * @param id the ID of the menu
     * @param dialogId the ID of the dialog which owns the menu
     * @param title the title of the menu or menu item
     * @param mnemonic the mnemonic for the menu or menu item
     * @param accelerator the accelerator for the menu or menu item
     * @param style the style from the dialog model
     * @param image the image from the dialog model
     * @param disabledImage the disabled image from the dialog model
     * @param menuType the type of the menu
     */
    public Menu(String id, String dialogId, String title, String mnemonic, String accelerator, String style, ImageInfo image,
    		ImageInfo disabledImage, MENU_TYPE menuType) {
    	super(id, dialogId, title, mnemonic, accelerator, style, image, disabledImage);
    	if (menuType == null) {
    		throw new IllegalArgumentException("menuType can not be null");
    	}
    	this.dialogInstance = 1;
    	this.type = menuType;
    }
    
    /**
     * Add a menu item to this menu.
     *
     * @param menuItem the menu item to add
     */
    public void addMenuItem(MenuItem menuItem) {
        addChild(menuItem);
    }

    /**
     * Add a separator to this menu.
     *
     * @param separator the separator to add
     */
    public void addSeparator(Separator separator) {
        addChild(separator);
    }

    /**
     * Add a menu to this menu.
     *
     * @param subMenu the menu to add
     */
    public void addMenu(Menu subMenu) {
        addChild(subMenu);
    }

    private <T extends MenuBase> void addChild(T child) {
        child.setParent(this);
        children.add(child);
    }

    /**
     * @return the dialogInstance
     */
    public int getDialogInstance() {
        return dialogInstance;
    }

    /**
     * @param dialogInstance the dialogInstance to set
     */
    public void setDialogInstance(int dialogInstance) {
        this.dialogInstance = dialogInstance;
    }

    /**
     * @return the menu type
     */
    public MENU_TYPE getType() {
        return type;
    }

    /**
     * @param type the menu type to set
     */
    public void setMenuType(MENU_TYPE type) {
        this.type = type;
    }

    /**
     * @return the children
     */
    public List<MenuBase> getChildren() {
        return children;
    }

    /**
     * @param children the children to set
     */
    public void setChildren(List<MenuBase> children) {
        this.children = children;
    }

    @Override
    public String toString() {
        return "Menu [id=" + getId() + "/" + getDialogId() + ", type=" + type + "]";
    }

}
