/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.table;

import java.text.Collator;
import java.util.Comparator;
import java.util.List;

import no.esito.log.Logger;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.Property;
import no.g9.os.AttributeConstant;

/**
 *  Comparator used to compare two list rows.
 * @param <T> actual list row class
 */
public class ListRowComparator<T extends ListRow> implements Comparator<T>,
        Comparable<ListRowComparator<T>> {

    private static final Logger log = Logger.getLogger(ListRowComparator.class);

    private Collator collator = Collator.getInstance();

    private final int initialPriority;

    /**
     * The column this comparator uses when sorting columns.
     */
    private final DialogObjectConstant column;

    /**
     * The sort direction of this comparator.
     */
    private Sorting direction = Sorting.ASCENDING;

    /** The next comparator in line. */
    private ListRowComparator<T> next;

    /**
     * Returns the next comparator in line.
     * @return the comparator after this
     */
    final ListRowComparator<T> getNextComparator() {
        return next;
    }
    
    /**
     * Creates a chain of comparators. When a list row comparator is comparing
     * two rows, it will invoke the next comparator (in the chain) if the
     * compared column values are equal.
     * 
     * @param <U>
     *            List row type
     * @param comparators
     *            the list of comparators
     * @return the first of the chained comparators
     */
    static <U extends ListRow> ListRowComparator<U> chainComparator(
            List<ListRowComparator<U>> comparators) {
        if (comparators == null || comparators.size() == 0) {
            return null;
        }

        for (int i = 0; i < comparators.size() - 1; i++) {
            comparators.get(i).next = comparators.get(i + 1);
        }
        // make sure last.next is null.
        comparators.get(comparators.size() - 1).next = null;

        return comparators.get(0);
    }
    
    /**
     * Creates a new ListRowComparator that will use values in the specified
     * column when comparing rows. Note that the column attribute must implement
     * the {@link Comparable} interface.
     * 
     * @param column
     *            column used when comparing rows
     * @param initialPriority
     *            the initial priority of this list row comparator
     * @throws IllegalArgumentException
     *             if column does not contain a {@link Comparable} attribute
     *             type
     */
    public ListRowComparator(DialogObjectConstant column, int initialPriority) {
        this.column = column;
        verifyColumn(column);
        this.initialPriority = initialPriority;
        collator.setStrength(Collator.IDENTICAL);
    }
    
    /**
     * Creates a new ListRowComparator that will use values in the specified
     * column when comparing rows. Note that the column attribute must implement
     * the {@link Comparable} interface.
     * 
     * @param column
     *            column used when comparing rows
     * @param initialPriority
     *            the initial priority of this list row comparator
     * @param sortDirection the initial sort direction of this list row comparator
     * @throws IllegalArgumentException
     *             if column does not contain a {@link Comparable} attribute
     *             type
     */
    public ListRowComparator(DialogObjectConstant column, int initialPriority, Sorting sortDirection) {
        this(column, initialPriority);
        setSorting(sortDirection);
    }

    private static void verifyColumn(DialogObjectConstant column) {
        AttributeConstant attribute = column.getAttribute();
        if (attribute == null) {
            return;
        }
        Class<?> attributeType = attribute.getAttributeType();

        if (!Comparable.class.isAssignableFrom(attributeType)) {
            String msg = null;
            msg = "Column " + column
                    + " does not contain java.lang.Comparable values";
            log.error("Cant create list row comparator for " + column + ":");
            throw new IllegalArgumentException(msg);
        }
    }

    /**
     * Enumerates sort direction.
     * 
     */
    public static enum Sorting implements Property<Enum<Sorting>> {
        /** All values are equal */
        NO_SORT(0),
        /** Ascending sort (A, B ... Z) */
        ASCENDING(1),
        /** Descending sort (Z, Y ... A) */
        DESCENDING(-1);

        private final int dir;

        Sorting(int dir) {
            this.dir = dir;
        }

        /**
         * Get the direction of this sorting, used when calculating the compare
         * result
         * 
         * @return the direction integer
         */
        int getDirection() {
            return dir;
        }

        /** The default sorting type which is ASCENDING */
        public final static Sorting DEFAULT = ASCENDING;

        @Override
        public Enum<Sorting> getDefaultValue() {
            return DEFAULT;
        }

        @Override
        public String toString() {
            return Sorting.class.getSimpleName() + ": " + name();
        }
    }

    /**
     * Toggles the sorting used when comparing list rows. 
     * <ul>
     * <li>NO_SORT -&gt; ASCENDING
     * <li>ASCENDING -&gt; DESCENDING
     * <li>DESCENDING -&gt; NO_SORT
     * </ul> 
     * 
     * @see #getSorting()
     */
    public void toggleSorting() {
        direction = getNext();
    }

    private Sorting getNext() {
        switch (direction) {
        case NO_SORT:
            return Sorting.ASCENDING;
        case ASCENDING:
            return Sorting.DESCENDING;
        case DESCENDING:
            return Sorting.NO_SORT;
        }
        // Should not happen.
        throw new RuntimeException("Unkonwn sort direction " + direction
                + ". Can't toggle!");
    }

    /**
     * Get the sorting used when comparing list rows.
     * 
     * @return the sort direction
     */
    public Sorting getSorting() {
        return direction;
    }

    /**
     * Set the sorting uses when comparing list rows.
     * 
     * @param sortDirection
     *            the sort direction
     */
    public void setSorting(Sorting sortDirection) {
        direction = sortDirection;
    }

    /**
     * Get the column used when comparing rows.
     * 
     * @return the sorting column
     */
    public DialogObjectConstant getColumn() {
        return column;
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    @Override
    public int compare(T o1, T o2) {

        int result = 0;

        if (direction != Sorting.NO_SORT) {
            Comparable v1 = (Comparable) o1.getValue(column);
            Comparable v2 = (Comparable) o2.getValue(column);

            if (v1 == null) { // Test for null
                result = (v2 == null) ? 0 : -1;
            } else if (v2 == null) {
                result = 1;
            } else if (v1 instanceof String) { // Locale compare of Strings
                result = collator.compare(v1, v2);
            } else {
                result = v1.compareTo(v2); // "Natural" ordering of comparables.
            }

            // apply direction
            result = result * direction.getDirection();
        }
        // Use next comparator if values in this column was equal.
        if (result == 0) {
            if (next != null) {
                result = next.compare(o1, o2);
            }
        }

        return result;

    }
    
    @Override
    public int hashCode() {
        final int prime= 31;
        int result= 1;
        result= prime * result + initialPriority;
        return result;
    }

    @SuppressWarnings("unchecked")
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ListRowComparator<T> other= (ListRowComparator<T>) obj;
        if (initialPriority != other.initialPriority)
            return false;
        return true;
    }

    @Override
    public int compareTo(ListRowComparator<T> o) {
        return initialPriority - o.initialPriority;
    }

    @Override
    public String toString() {
        return "ListRowComparator [column=" + column + ", direction="
                + direction + ", next=" + next + "]";
    }

    
    
}
