/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.table;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import no.esito.log.Logger;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.ListRow;
import no.g9.os.AttributeConstant;

/**
 * ColumnFilter is used to filter <code>ListRow</code>s based on a filter value
 * and filter type.
 * 
 * @param <U> the filter type class
 * @param <T> the generated ListRow class
 */
public abstract class RowFilter<U, T extends ListRow> {

    /** Logger */
    private static final Logger log = Logger.getLogger(RowFilter.class);

    private final DialogObjectConstant column;
    private U filterValue;

    /**
     * Create a new row filter. The filter will use the value in the specified
     * column to determine if the row passes this filter.
     * 
     * @param column the column used when filtering
     * @param filterValue the filter value
     */
    protected RowFilter(DialogObjectConstant column, U filterValue) {
        this.column = column;
        this.filterValue = filterValue;
        paramCheck(column, filterValue);
    }

    private void paramCheck(DialogObjectConstant column, U filterValue) {
        String msg = null;
        if (column == null) {
            msg = "Column";
        } else if (filterValue == null) {
            msg = "filterValue";
        }
        if (msg != null) {
            msg += " cannot be null!";
            throw new IllegalArgumentException(msg);
        }
    }

    /**
     * Specifies the criteria used by a Filter.
     * 
     */
    public enum FilterType {
        /**
         * Accept values that are less than the filter value.
         */
        LESS_THAN,

        /** Accept values that are greater than the filter value. */
        GREATER_THAN,

        /** Accept values that are equal to the filter value. */
        EQUAL,

        /** Accept values that are not equal to the filter value. */
        NOT_EQUAL;
    }

    /**
     * Test if the specified row passes this filter. This method returns
     * <code>true</code> if the row is not caught by this filter.
     * 
     * @param row ListRow to test
     * @return <code>true</code> if the row is accepted by this filter
     */
    public abstract boolean passFilter(T row);

    /**
     * Get the value used by this filter when checking if a row should be
     * included.
     * 
     * @return the filter value
     */
    public final U getFilterValue() {
        return filterValue;
    }

    /**
     * Set the value used by this filter when checking if a row should be
     * included.
     * 
     * @param filterValue the filter value
     */
    public void setFilterValue(U filterValue) {
        this.filterValue = filterValue;
    }

    /**
     * Get the column this filter applies to.
     * 
     * @return the dialog object constant denoting the column
     */
    public DialogObjectConstant getColumn() {
        return column;
    }

    /**
     * Create a filter that uses a java.lang.Comparable filter value. Note that
     * <code>null</code> values will never pass this filter.
     * 
     * @param column the column this filter applies to
     * @param filterValue the filter value
     * @param filterType the type of filter
     * @param <U> the filter type class
     * @param <T> the generated ListRow class
     * @return a filter that uses <code>Comparable</code> when filtering
     */
    public static <U extends Comparable<U>, T extends ListRow> RowFilter<U, T> comparableFilter(
            DialogObjectConstant column, U filterValue, FilterType filterType) {
        return new ComparableFilter<U, T>(column, filterValue, filterType);
    }

    /**
     * Create a filter that uses the a regular expression to filter values.
     * 
     * @param <T> the generated ListRow class
     * @param column the column this filter applies to
     * @param regexp the regular expression
     * @return a filter that uses the regular expression when filtering
     */
    public static <T extends ListRow> RowFilter<String, T> regexpFilter(
            DialogObjectConstant column, String regexp) {
        return new RegexpFilter<T>(column, regexp);
    }

    /**
     * Implementation of a regular expression filter.
     * 
     * @param <T> the list row implementation
     */
    private static class RegexpFilter<T extends ListRow> extends
            RowFilter<String, T> {

        private Matcher matcher;

        private RegexpFilter(DialogObjectConstant column, String regexp) {
            super(column, regexp);
            updateMathcer(regexp);
        }

        private void updateMathcer(String regexp) {
            Pattern pattern = Pattern.compile(regexp);
            matcher = pattern.matcher("");
        }

        @Override
        public boolean passFilter(T row) {
            Object obj = row.getValue(getColumn());

            if (obj == null) {
                obj = "";
            }

            String value = String.valueOf(obj);

            matcher.reset(value);
            return matcher.find();
        }

        @Override
        public void setFilterValue(String filterValue) {
            super.setFilterValue(filterValue);
            updateMathcer(filterValue);
        }

    }

    /**
     * Implementation of a comparable filter.
     * 
     * @param <U> the filter type class
     * @param <T> the list row implementation
     */
    private static class ComparableFilter<U, T extends ListRow> extends
            RowFilter<U, T> {
        private final FilterType filterType;

        private ComparableFilter(DialogObjectConstant column, U filterValue,
                FilterType filterType) {
            super(column, filterValue);
            paramCheck(column);
            this.filterType = filterType;
        }

        /**
         * Test that the specified column contains values that have a "natural"
         * ordering - viz. that they implement the Comparable interface.
         * 
         * @param column the column to check
         * @throws RuntimeException if the column does not contain a Comparable
         *             value
         */
        private void paramCheck(DialogObjectConstant column)
                throws RuntimeException {
            AttributeConstant attribute = column.getAttribute();
            Class<?> attributeType = attribute.getAttributeType();

            if (!Comparable.class.isAssignableFrom(attributeType)) {
                String msg = null;
                msg = "Column " + column + " does not contain java.lang.Comparable values";
                log.error("Cant create comparable filter for " + column + ":");
                throw new IllegalArgumentException(msg);
            }
        }

        // parameters are checked when creating the filter.
        @SuppressWarnings("unchecked")
        @Override
        public boolean passFilter(T row) {
            Comparable<U> value = (Comparable<U>) row.getValue(getColumn());
            log.trace("Testing " + row + " column " + getColumn() + " against "
                    + getFilterValue() + " using " + filterType + " filter");
            if (value == null) {
                return false;
            }
            boolean pass = false;
            switch (filterType) {
            case EQUAL:
                pass = value.compareTo(getFilterValue()) == 0;
                break;
            case NOT_EQUAL:
                pass = value.compareTo(getFilterValue()) != 0;
                break;
            case GREATER_THAN:
                pass = value.compareTo(getFilterValue()) > 0;
                break;
            case LESS_THAN:
                pass = value.compareTo(getFilterValue()) < 0;
                break;
            default:
                pass = false;
            }
            log.trace((pass ? "passed " : "did not pass "));
            return pass;
        }

    }

}
