/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.table;

import java.io.PrintStream;
import java.util.Collection;
import java.util.List;
import java.util.NoSuchElementException;

import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.Property;
import no.g9.client.core.view.PropertyManager;
import no.g9.client.core.view.table.ListRowComparator.Sorting;

/**
 * The general table model used by jVine targets. The table model is used to
 * hold all the data that can be displayed by a table in addition to various
 * properties concerning rows, columns and table cells.
 * <p>
 * {@link ListRow} is used to represents rows. Rows are referenced using a
 * zero-based <em>rowIndex</em>. When setting a property, the property is
 * propagated to the ListRow's {@link PropertyManager}.
 *
 * <p>
 * The table model maintains two lists of ListRows:
 * <ul>
 * <li>The <em>table data</em> list is the complete list of ListRows in this
 * table model. All methods that uses rowIndex as a parameter or return value
 * relates to the table data list. This list can be accessed by means of the
 * {@link #getTableData()} method.
 * <li>The <em>table view</em> list is the filtered and sorted list of ListRows
 * that can be presented by a dialog. This list can be accessed by means of the
 * {@link #getTableView()} method.
 * </ul>
 *
 * @param <T>
 *            The actual implementing ListRow class
 *
 */
public interface TableModel<T extends ListRow> {

    /**
     * Enumerates the various row selection models.
     */
    enum SelectionModel implements Property<Enum<SelectionModel>> {

        /** No row selection is possible. */
        NO_SELECT,

        /** At most one row can be selected */
        SINGLE_SELECT,

        /** Selection of several rows is possible */
        MULTI_SELECT;

        @Override
        public Enum<SelectionModel> getDefaultValue() {
            return DEFAULT;
        }

        /** The default selection model which is SINGLE_SELECT */
        public static final SelectionModel DEFAULT = SINGLE_SELECT;

        @Override
        public String toString() {
            return SelectionModel.class.getSimpleName() + ": " + name();
        }
    }

    /**
     * Set the list of ListRowComparators to use when sorting this table model.
     * The sequence of the list will determine the initial sorting.
     *
     * @param comparatorList
     *            list of comparators to use, one for each sortable column
     */
    public void setListRowComparator(List<ListRowComparator<T>> comparatorList);

    /**
     * Toggle the sorting used by the first sorting column.
     *
     * @see ListRowComparator#toggleSorting()
     */
    void toggleSorting();

    /**
     * Get the sort direction used by the first sorting column.
     *
     * @return the sort direction
     *
     * @see ListRowComparator#getSorting()
     */
    Sorting getSorting();

    /**
     * Set the sort direction used by the first sorting column.
     *
     * @param sortDirection
     *            the sort direction
     *
     * @see ListRowComparator#setSorting(Sorting)
     */
    void setSorting(Sorting sortDirection);

    /**
     * Use the specified column first when comparing list rows.
     *
     * @param column
     *            first column to use when sorting
     */
    void setFirstSortingColumn(DialogObjectConstant column);

    /**
     * Sort the table view. The table view is sorted according to the defined
     * ListRow comparators.
     *
     * @see #setListRowComparator(List)
     */
    void sortTableView();

    /**
     * Get the column that is first in the sorting sequence.
     *
     * @return the column that is first in the sorting
     */
    DialogObjectConstant getFirstSortingColumn();

    /**
     * Add the row filter to this table model. The <em>table view</em> list will
     * be updated.
     *
     * @param rowFilter
     *            row filter(s) to add
     */
    public void addRowFilter(RowFilter<?, ListRow> rowFilter);

    /**
     * Add the row filters to this table model. The <em>table view</em> list
     * will be updated.
     *
     * @param rowFilters
     *            row filter(s) to add
     */
    public void addRowFilters(Collection<RowFilter<?, ListRow>> rowFilters);

    /**
     * Remove the row filter from this table model. The
     * <em>table view</em> list will be updated.
     *
     * @param rowFilter
     *            row filter(s) to remove
     */
    public void removeRowFilter(RowFilter<?, ListRow> rowFilter);

    /**
     * Remove the row filters from this table model. The
     * <em>table view</em> list will be updated.
     *
     * @param rowFilters
     *            row filters to remove
     */
    public void removeRowFilters(Collection<RowFilter<?, ListRow>> rowFilters);

    /**
     * Removes all row filters used by this table model.
     */
    void clearRowFilters();

    /**
     * Get the registered row filters for this table model.
     *
     * @return an unmodifiable view of the row filters collection
     */
    Collection<RowFilter<?, ListRow>> getRowFilters();

    /**
     * Update the table view list according to the defined filters.
     */
    public void filterTableView();

    /**
     * Get the table data list. This is the list used for maintaining the
     * unsorted and un-filtered table list rows.
     *
     * <p>
     * <b>Note</b> that any changes made directly to this list will be detected
     * by the table model and trigger and update of the table view list.
     *
     * @return table data list
     */
    public List<T> getTableData();

    /**
     * Get the table view list. This is the sorted and filtered list presented
     * to dialog views.
     *
     * <p>
     * <b>Note</b> that any changes made directly to this list will not be
     * detected by the table model. Thus adding or removing rows to this list
     * might not yield the desired result.
     *
     * @return table view list
     */
    public List<T> getTableView();

    /**
     * Set the table's selection model.
     *
     * @param selectionModel
     *            the selection model for the table
     */
    void setSelectionModel(SelectionModel selectionModel);

    /**
     * Get the tables selection model.
     *
     * @return the
     */
    SelectionModel getSelectionModel();

    // ////////////////////////////////////////////////////////
    // /////////////// Cell value methods //////////////
    // ////////////////////////////////////////////////////////

    /**
     * Get the cell value at the specified row and column.
     *
     * @param rowIndex
     *            table data's index of the row
     * @param columnIdentifier
     *            constant denoting the column
     * @return the cell value
     * @throws IndexOutOfBoundsException
     *             if the specified row index is out of range (
     *             <code>rowIndex &lt; 0 || rowIndex &gt; getRowCount()</code>)
     * @throws NoSuchElementException
     *             if the specified column is not part of this table data
     *
     */
    Object getValueAt(int rowIndex, DialogObjectConstant columnIdentifier);

    /**
     * Set the specified cell value.
     * <p>
     * <b>Note:</b> The table view is <em>not</em> updated as a result of
     * invoking this method.
     *
     * @param rowIndex
     *            table data's index of the row
     * @param columnIdentifier
     *            constant denoting the column
     * @param value
     *            cell value
     * @see #filterTableView()
     * @throws IndexOutOfBoundsException
     *             if the specified row index is out of range (
     *             <code>rowIndex &lt; 0 || rowIndex &gt; getRowCount()</code>)
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model data
     */
    void setValueAt(int rowIndex, DialogObjectConstant columnIdentifier,
            Object value);

    // ////////////////////////////////////////////////////////
    // //////////////// row and column counts /////////////////
    // ////////////////////////////////////////////////////////
    /**
     * Get the number of rows in this table's data list.
     *
     * @return the data list's row count
     */
    int getTableDataRowCount();

    /**
     * Get the number of rows in this table's view list.
     *
     * @return the view list's row count
     */
    int getTableViewRowCount();

    /**
     * Get the number of columns in this table model.
     *
     * @return the column count
     */
    int getColumnCount();

    /**
     * Get the index of the specified row. The table's data list are compared
     * with the specified ListRow, and the index of the first row that matches
     * (using the equals method) is returned. If no match is found,
     * <code>-1</code> is returned.
     *
     * @param row
     *            row to get index for
     * @return the table data's row index or -1 if the table data list does not
     *         contain the specified ListRow
     * @throws NullPointerException
     *             if the specified row is <code>null</code>
     */
    int indexOf(T row);

    /**
     * Transform the given tableViewIndex to the corresponding table model
     * index. This method is used to get from a displayed row index to the row's
     * index in the table data.
     *
     * @param tableViewIndex
     *            displayed row index
     * @return the row's table data index
     */
    int toTableDataIndex(int tableViewIndex);

    // ////////////////////////////////////////////////////////
    // ///////////////adding and removing rows/////////////////
    // ////////////////////////////////////////////////////////
    /**
     * Get the list row at the specified table data's index (zero-based).
     *
     * @param rowIndex
     *            the table data's index of the row to get
     * @return the row at the specified index
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             <code>rowIndex &lt; 0 || rowIndex %gt;= getRowCount()</code>)
     */
    T get(int rowIndex);

    /**
     * Inserts a row or rows at the specified position in the table data list .
     * Shifts the row at that position (if any) an any subsequent rows down.
     *
     * @param rowIndex
     *            table data's index at which the specified row is to be
     *            inserted
     * @param row
     *            row to be inserted
     * @throws IndexOutOfBoundsException
     *             if the specified row index is out of range (
     *             <code>rowIndex &lt; 0 || rowIndex &gt; getTableDataRowCount()</code>
     *             )
     * @throws NullPointerException
     *             if the specified row is <code>null</code>
     *
     * @throws IllegalArgumentException
     *             if the rows does not match the column definitions
     */
    @SuppressWarnings("unchecked")
	void insert(int rowIndex, T... row);

    /**
     * Inserts a row or rows at the specified position in the table data list.
     * Shifts the row at that position (if any) an any subsequent rows down. The
     * internal ordering of the inserted rows are determined by the iteration
     * order of the <code>rows</code> collection.
     *
     * @param rowIndex
     *            table data's index at which the specified row is to be
     *            inserted
     * @param rows
     *            rows to be inserted
     * @throws IndexOutOfBoundsException
     *             if the specified row index is out of range (
     *             <code>rowIndex &lt; 0 || rowIndex &gt; getTableDataRowCount()</code>
     *             )
     * @throws NullPointerException
     *             if the specified row is <code>null</code>
     * @throws IllegalArgumentException
     *             if the rows does not match the column definitions
     */
    void insert(int rowIndex, Collection<T> rows);

    /**
     * Appends the specified rows to this table data's list.
     *
     * @param row
     *            the row or rows to append
     * @throws NullPointerException
     *             if rows is <code>null</code> or contains a <code>null</code>
     *             reference
     * @throws IllegalArgumentException
     *             if the rows does not match the column definitions
     */
    @SuppressWarnings("unchecked")
	void add(T... row);

    /**
     * Appends the specified rows to this table data's list.
     *
     * @param rows
     *            the rows append
     * @throws NullPointerException
     *             if rows is <code>null</code> or contains a <code>null</code>
     *             reference
     * @throws IllegalArgumentException
     *             if the rows does not match the column definitions
     */
    void add(Collection<T> rows);

    /**
     * Removes the row at the specified table data index. Subsequent rows (if
     * any) are shifted up.
     *
     * @param rowIndex
     *            table data's index of the row to remove.
     * @return the removed row
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             <code>rowIndex &lt; 0 || rowIndex &gt;= getRowCount()</code>)
     */
    T remove(int rowIndex);

    /**
     * Removes specified row or rows from the table data list. Subsequent rows
     * (if any) are shifted up.
     *
     * @param row
     *            the row or rows to remove.
     * @return <code>true</code> if the table data changed as a result of the
     *         call
     */
    @SuppressWarnings("unchecked")
	boolean remove(T... row);

    /**
     * Removes specified row or rows from the table data list. See
     * {@link #remove(ListRow...)}
     *
     * @param rows
     *            the rows to remove.
     * @return <code>true</code> if the table data changed as a result
     *         of the call
     */
    boolean removeAll(Collection<T> rows);

    /**
     * Clears this table model by removing all ListRows.
     */
    void clear();

    /**
     * Test if this table model contains any rows.
     *
     * @return <code>true</code> if this table model is empty (
     *         {@link #getTableDataRowCount()} == 0)
     */
    boolean isEmpty();

    // ////////////////////////////////////////////////////////
    // /////////// SELECTING ROWS /////////////////////////
    // ////////////////////////////////////////////////////////
    /**
     * Set the selected property of the specified row.
     *
     * @param rowIndex
     *            table data's index of the row
     * @param selected
     *            if <code>true</code> the row will be selected
     * @see #getSelectionModel()
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     * @throws IllegalStateException
     *             if the selection model prohibits the selection
     */
    void setSelected(int rowIndex, boolean selected);

    /**
     * Set the selected property of the specified row.
     *
     * @param row
     *            the row to select
     *
     * @param selected
     *            if <code>true</code> the row will be selected
     * @see #getSelectionModel()
     * @throws IllegalStateException
     *             if the selection model prohibits the selection
     */
    void setSelected(T row, boolean selected);

    /**
     * Set the selected property of all rows.
     *
     * @see #getSelectionModel()
     * @param selected
     *            if <code>true</code> the row will be selected
     * @throws IllegalStateException
     *             if the selection model prohibits the selection
     */
    void setSelected(boolean selected);

    /**
     * Test if the specified row is selected.
     *
     * @param rowIndex
     *            table data's index of the row to test
     * @return <code>true</code> if the specified row <em>is</em> selected.
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     */
    boolean isSelected(int rowIndex);

    /**
     * Get the list of selected rows. The list maintains the internal original
     * ordering of the selected rows.
     *
     * @return a list of selected rows.
     */
    List<T> getSelected();

    /**
     * Get the number of selected rows.
     *
     * @return number of selected rows
     */
    int getSelectionCount();

    // ////////////////////////////////////////////////////////
    // //////// ENABLED PROPERTY ///////////
    // ////////////////////////////////////////////////////////

    /**
     * Set the enabled property for all cells in the entire table.
     *
     * @param enabled
     *            if <code>true</code>, all cells in the row will be enabled
     * @see #isCellEnabled(int, DialogObjectConstant)
     */
    void setEnabled(boolean enabled);

    /**
     * Set the enabled property for all cells in the specified row.
     *
     * @param rowIndex
     *            table data's index of the row to set editable
     * @param enabled
     *            if <code>true</code>, all cells in the row will be enabled
     * @see #isCellEnabled(int, DialogObjectConstant)
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     */
    void setRowEnabled(int rowIndex, boolean enabled);

    /**
     * Set the enabled property for all cells in the specified row.
     *
     * @param row
     *            the row to set editable
     * @param enabled
     *            if <code>true</code>, all cells in the row will be enabled
     * @see #isCellEnabled(int, DialogObjectConstant)
     */
    void setRowEnabled(T row, boolean enabled);

    /**
     * Set the enabled property for all cells in the specified column.
     *
     * @param columnIdentifier
     *            constant denoting the column
     * @param enabled
     *            if <code>true</code>, all cells in the column will be enabled
     * @see #isCellEnabled(int, DialogObjectConstant)
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     */
    void setColumnEnabled(DialogObjectConstant columnIdentifier, boolean enabled);

    /**
     * Set the enabled property the specified cell.
     *
     * @param rowIndex
     *            table data's row index
     * @param columnIdentifier
     *            constant denoting the column
     * @param enabled
     *            if <code>true</code>, the cells will be editable
     * @see #isCellEnabled(int, DialogObjectConstant)
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     */
    void setCellEnabled(int rowIndex, DialogObjectConstant columnIdentifier,
            boolean enabled);

    /**
     * Get the enabled property value of the specified cell. If the enabled
     * property of a cell is <code>false</code> the cell will not be able to
     * receive focus. If the cell is <em>visible</em> and <em>enabled</em> the
     * cell will be able to receive focus.
     *
     * @param rowIndex
     *            table data's row index
     * @param columnIdentifier
     *            constant denoting the column
     * @return <code>true</code> if the cell is enabled
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     *
     */
    boolean isCellEnabled(int rowIndex, DialogObjectConstant columnIdentifier);

    // ////////////////////////////////////////////////////////
    // ////// SHOW PROPERTY /////////////////////////////
    // ////////////////////////////////////////////////////////

    /**
     * Set the shown property for all cells in the table.
     *
     * @param shown
     *            if <code>true</code>, all cells in the row will be rendered
     * @see #isCellShown(int, DialogObjectConstant)
     */
    void setShown(boolean shown);

    /**
     * Set the shown property for all cells in the specified row.
     *
     * @param rowIndex
     *            table data's row index
     * @param shown
     *            if <code>true</code>, all cells in the row will be rendered
     * @see #isCellShown(int, DialogObjectConstant)
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     */
    void setRowShown(int rowIndex, boolean shown);

    /**
     * Set the shown property for all cells in the specified row.
     *
     * @param row
     *            the row to set shown
     * @param shown
     *            if <code>true</code>, all cells in the row will be rendered
     * @see #isCellShown(int, DialogObjectConstant)
     */
    void setRowShown(T row, boolean shown);

    /**
     * Set the shown property for all cells in the specified column.
     *
     * @param columnIdentifier
     *            constant denoting the column
     * @param shown
     *            if <code>true</code>, all cells in the column will be rendered
     * @see #isCellEnabled(int, DialogObjectConstant)
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     */
    void setColumnShown(DialogObjectConstant columnIdentifier, boolean shown);

    /**
     * Set the shown property the specified cell.
     *
     * @param rowIndex
     *            table data's row index
     * @param columnIdentifier
     *            constant denoting the column
     * @param shown
     *            if <code>true</code>, the cell will be rendered
     * @see #isCellEnabled(int, DialogObjectConstant)
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     */
    void setCellShown(int rowIndex, DialogObjectConstant columnIdentifier,
            boolean shown);

    /**
     * Get the shown property value of the specified cell. If the shown property
     * of a cell is <code>true</code> the cell will be rendered.
     *
     * @param rowIndex
     *            table data's row index
     * @param columnIdentifier
     *            constant denoting the column
     * @return <code>true</code> if the cell is shown
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     *
     */
    boolean isCellShown(int rowIndex, DialogObjectConstant columnIdentifier);

    // ////////////////////////////////////////////////////////
    // ///////GENERAL PROPERTY ////
    // ////////////////////////////////////////////////////////

    /**
     * Set the specified property on all cells in the table.
     *
     * @param <U>
     *            The property value type
     *
     * @param property
     *            the property
     * @param propertyValue
     *            value of the property
     */
    <U> void setProperty(Property<U> property, U propertyValue);

    /**
     * Set the specified property on all cells in the row.
     *
     * @param <U>
     *            The property value type
     * @param rowIndex
     *            table data's row index
     * @param property
     *            the property
     * @param propertyValue
     *            value of the property
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     *
     */
    <U> void setRowProperty(int rowIndex, Property<U> property, U propertyValue);

    /**
     * Set the specified property on all cells in the row.
     *
     * @param <U>
     *            The property value type
     * @param row
     *            table row to set the property on
     * @param property
     *            the property
     * @param propertyValue
     *            value of the property
     */
    <U> void setRowProperty(T row, Property<U> property, U propertyValue);

    /**
     * Set the specified property on all cells in the column.
     *
     * @param <U>
     *            The property value type
     * @param columnIdentifier
     *            constant denoting the column
     * @param propertyName
     *            name identifying the property
     * @param value
     *            value of the property
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     */
    <U> void setColumnProperty(DialogObjectConstant columnIdentifier,
            Property<U> propertyName, U value);

    /**
     * Set the specified property on the specified cell.
     *
     * @param <U>
     *            The property value type
     * @param rowIndex
     *            table data's row index
     * @param columnIdentifier
     *            constant denoting the column
     * @param propertyName
     *            name identifying the property
     * @param propertyValue
     *            value of the property
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     */
    <U> void setCellProperty(int rowIndex,
            DialogObjectConstant columnIdentifier, Property<U> propertyName,
            U propertyValue);

    /**
     * Get the property value for the specified cell.
     *
     * @param <U>
     *            The property value type
     * @param rowIndex
     *            table data's row index
     * @param columnIdentifier
     *            constant denoting the column
     * @param propertyName
     *            name identifying the property
     * @return value of the property
     * @throws IndexOutOfBoundsException
     *             if the specified index is out of range (
     *             {@code rowIndex < 0 || rowIndex >= getTableDataRowCount()}
     *             )
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     *
     */
    <U> U getCellProperty(int rowIndex, DialogObjectConstant columnIdentifier,
            Property<U> propertyName);

    /**
     * Debug print the table contents to specified print stream. <b>OBS: This
     * method should <em>not</em> be invoked in production code.</b>
     *
     * <p>
     * This method is intended for <em>debug</em> purposes only and should be
     * considered as an advanced variant of the general <code>toString()</code>
     * method. The exact format of the printed text may vary between releases
     * (even patches).
     *
     * @param out
     *            the print stream to write to.
     */
    void prettyPrint(PrintStream out);

}
