/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.spreadsheet;

import java.io.File;
import java.io.OutputStream;

import no.g9.client.core.controller.DialogConstant;
import no.g9.client.spreadsheet.WorkbookProvider.WORKBOOK_FORMAT;
import no.g9.support.ClientContext;
import no.g9.support.ObjectFactory;
import no.g9.support.ObjectSelection;
import no.g9.support.Registry;

/**
 * The default implementation of the SpreadsheetService interface.
 * It will use a generated G9ExcelExporter subclass for the spreadsheet export.
 */
public class DefaultSpreadsheetService implements SpreadsheetService {

	/** The Excel format to use in the export. The default is XLSX. */
	private WORKBOOK_FORMAT format = WORKBOOK_FORMAT.XLSX;

	/** The ClientContext parameter name for the Excel format to use. */
	public static final String FORMAT_PARAM = "WorkbookFormat";

	private static final String EXPORTER_SUFFIX = "ExcelExporter";

	private static final String SPREADSHEET_EXPORTER = ".spreadsheetExporter";

	@Override
	public SpreadsheetExporter exportToSpreadsheet(ObjectSelection os, DialogConstant targetDialog, File file) {
		G9ExcelExporter ee = getExporter(os, targetDialog);
		ee.createBook(os, file, format);
		return ee;
	}

	@Override
	public SpreadsheetExporter exportToSpreadsheet(ObjectSelection os, DialogConstant targetDialog, OutputStream outputStream) {
		G9ExcelExporter ee = getExporter(os, targetDialog);
		ee.createBook(os, outputStream, format);
		return ee;
	}

	@Override
	public SpreadsheetExporter exportToSpreadsheet(ObjectSelection os, DialogConstant targetDialog, File file, ClientContext ctx) {
		parseParams(ctx);
		G9ExcelExporter ee = getExporter(os, targetDialog);
		ee.createBook(os, file, format);
		return ee;
	}

	@Override
	public SpreadsheetExporter exportToSpreadsheet(ObjectSelection os, DialogConstant targetDialog, OutputStream outputStream, ClientContext ctx) {
		parseParams(ctx);
		G9ExcelExporter ee = getExporter(os, targetDialog);
		ee.createBook(os, outputStream, format);
		return ee;
	}

	@Override
	public String getFileSuffix() {
		return format.getSuffix();
	}

	@Override
	public String getMimeType() {
		return format.getMimeType();
	}

	/**
	 * Check if the client context contains any parameters used by
	 * the Excel exporter.
	 * <br>
	 * The Excel exporter can generate exports with
	 * both XLS and XLSX formats. The default is XLSX, but XLS
	 * will be used if the FORMAT_PARAM is set to <code>WORKBOOK_FORMAT.XLS</code>.
	 * <br>
	 * If the SpreadsheetService was called from a generated
	 * ExportToSpreadsheet action, parameters can be added to the
	 * client context in the contextHook method of an ActionHookAdapter.
	 * <br>
	 * This method can be overridden in a subclass if using a spreadsheet
	 * exporter which supports other parameters.
	 *
	 * @param ctx The client context object
	 */
	protected void parseParams(ClientContext ctx) {
		if (ctx != null) {
			WORKBOOK_FORMAT ctxFormat = (WORKBOOK_FORMAT) ctx.getParam(FORMAT_PARAM);
			if (ctxFormat != null) {
				format = ctxFormat;
			}
		}
	}

	/**
	 * Get the Excel exporter to use for the spreadsheet export.
	 * <br>
	 * The default is to first check if there is a property file named
	 * "ApplicationName".spreadsheetExporter with a "DialogName" property.
	 * If found, the property value should be the class name of the exporter.
	 * <br>
	 * Else, the property ServiceApplication."ApplicationName".DefaultSpreadsheetExporterPackage
	 * is used as the package prefix for the exporter class, appended with the package
	 * of the dialog and the "DialogName"ExcelExporter to get the exporter class name.
	 * <br>
	 * This method can be overridden in a subclass if another spreadsheet exporter lookup
	 * method is wanted.
	 *
	 * @param os The object selection, with the name of the object selection
	 * @param targetDialog The dialog to be exported
	 * @return The Excel exporter to use for the spreadsheet export
	 */
	protected G9ExcelExporter getExporter(ObjectSelection os, DialogConstant targetDialog) {
		String exporterName = getExporterName(os, targetDialog);
		return (G9ExcelExporter) ObjectFactory.newObject(exporterName);
	}

	private String getExporterName(ObjectSelection os, DialogConstant targetDialog) {
		String exporterName = null;
		String propFileName = os.getApplicationName() + SPREADSHEET_EXPORTER;
		Registry reg = Registry.getRegistry();
		String dialogName = targetDialog.getG9Name();

		exporterName = reg.getApplicationProperty(propFileName, dialogName);
		if (exporterName == null) {
			String propName = Registry.SRV_APP_PRE + "." + os.getApplicationName() + "." + Registry.DEFAULT_SPREADSHEET_PACKAGE;
			exporterName = reg.getG9Property(propName) + "." + dialogName.toLowerCase() + "." + dialogName + EXPORTER_SUFFIX;
		}
		return exporterName;
	}

}
