/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.action;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import no.esito.jvine.controller.JVineAppController;
import no.esito.log.Logger;
import no.g9.client.core.controller.DialogController;

/**
 * Blocks the invoking thread. The monitor is "this".
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public class BlockingActionMethodRunner implements ActionMethodRunner {

    private final static Logger log = Logger
            .getLogger(BlockingActionMethodRunner.class);

    private ActionQueue guiActionQueue;
    // = new BlockingGuiActionQueue();

    private final ThreadManager threadManager = new ThreadManager();

    /**
     * An event dispatcher that blocks and wait immediately after posting the
     * runnable event method to a single threaded executor queue.
     */
    private ExecutorService singleThreadedExecutor = null;

    private DialogController dialogController;

    private ExecutorService getExecutor() {
        if (singleThreadedExecutor == null
                || singleThreadedExecutor.isShutdown()) {
            if (log.isTraceEnabled()) {
                String msg = "Creating a new method executor.";
                if (singleThreadedExecutor != null) {
                    msg = "Previous executor was shut down. " + msg;
                }
                log.trace(msg);
            }
            singleThreadedExecutor = Executors
                    .newSingleThreadExecutor(threadManager);
        }
        return singleThreadedExecutor;
    }

    @Override
    public synchronized void shutDown() {
        if (singleThreadedExecutor != null) {
            if (log.isDebugEnabled()) {
                log.debug("Shutting down executor " + threadManager);
            }
            singleThreadedExecutor.shutdown();
        }
    }

    @Override
    public void doMethod(Runnable eventMethod) {
        if (log.isDebugEnabled()) {
            log.debug("Submitting action method for execution");
        }

        ExecutorService service = getExecutor();
        service.execute(eventMethod);

        getActionQueue().ready();
    }

    @Override
    public void setThreadNamePrefix(String threadName) {
        threadManager.setThreadNamePrefix(threadName);
    }

    @Override
    public synchronized ActionQueue getActionQueue() {
        if (guiActionQueue == null) {
            JVineAppController appCtrl = JVineAppController
                    .getInstance(dialogController.getApplicationController());
            guiActionQueue = appCtrl.getActionQueue();
        }
        return guiActionQueue;
    }

    @Override
    public void releaseGui() {
        getActionQueue().release();
    }

    @Override
    public <V> V invokeOnGui(Callable<V> guiMethod)
            throws InvocationTargetException {
        return getActionQueue().perform(guiMethod);
    }

    @Override
    public void invokeOnGui(Runnable guiMethod)
            throws InvocationTargetException {
        getActionQueue().perform(guiMethod);
    }

    /*
     * (non-Javadoc)
     * @see
     * no.esito.jvine.action.ActionMethodRunner#setDialogController(no.g9
     * .client.core.controller.DialogController)
     */
    @Override
    public synchronized void setDialogController(DialogController controller) {
        this.dialogController = controller;
    }

}