/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.communication;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import no.g9.client.core.communication.G9ActionPayload;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogSetupValue;

import com.fasterxml.jackson.core.JsonGenerationException;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Utilities for handling messages
 *
 */
public class SystemMessageUtils {

    /**
     * Creates a list of SystemMessages from a JSON.
     *
     * @param inputMessagesInJson the source JSON
     * @return list of messages
     */
    public static List<SystemMessage> fromJSon(String inputMessagesInJson) {
        ObjectMapper mapper = new ObjectMapper();
        List<SystemMessage> decodedMessages = new ArrayList<SystemMessage>();
        try {
            if (inputMessagesInJson != null) {
                List<String> codedMessages = mapper.readValue(inputMessagesInJson, new TypeReference<List<String>>() { /* EMPTY */ });

                for (String codedMessage : codedMessages) {
                    decodedMessages.add(new SystemMessage(codedMessage));
                }
            }
        } catch (JsonParseException e) {
            // This means someone is experimenting with the validity of the parser
            e.printStackTrace();
        } catch (JsonMappingException e) {
            // This means someone is experimenting with the validity of the parser
            e.printStackTrace();
        } catch (IOException e) {
            // No io is used, so this should never happen
            e.printStackTrace();
        }
        return decodedMessages;
    }

    /**
     * Converts a list of messages to a JSON representation.
     *
     * @param inputMessages a list of messages
     * @return the JSON representation
     */
    public static String toJson(List<SystemMessage> inputMessages) {
        ArrayList<String> stringList = new ArrayList<String>();
        for (SystemMessage systemMessage : inputMessages) {
            stringList.add(systemMessage.code());
        }

        ObjectMapper mapper = new ObjectMapper();
        StringWriter writer = new StringWriter();

        try {
            mapper.writeValue(writer, stringList);
            return writer.getBuffer().toString();
        } catch (JsonGenerationException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (JsonMappingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return "";
    }

    /**
     * Creates a message to a specific dialog.
     *
     * @param receiver the receiver
     * @param target the port
     * @param payload the payload
     * @return the message
     */
   public static SystemMessage createSystemMessage(String receiver, DialogConstant target, String payload) {
       return new SystemMessage(receiver, target.getInternalName(), payload);
   }

    /**
     * Codes a setup value.
     *
     * @param setupValue setup value to be coded
     * @return the coded setup value
     */
    public static String codeSetupValue(Object setupValue) {
        if(setupValue instanceof String){
            return (String)setupValue;
        }
        return "";
    }

    /**
     * Extracts the action setup value from a g9 action message.
     *
     * @param message the message
     * @param applicationController the application controller which handles the call back
     * @return the setup value
     */
    public static DialogSetupValue<?> extractActionSetupValue(SystemMessage message, ApplicationController applicationController) {
        G9ActionPayload actionPayload=new G9ActionPayload(message.payload);
        if (!actionPayload.payload.isEmpty()) {
            return new SystemMessageDialogSetupValue(actionPayload.payload, applicationController);
        }
        return null;
    }

}
