/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import no.esito.jvine.model.CurrentRoleObject;
import no.esito.jvine.model.DisplayableOSRole;
import no.esito.jvine.model.TreeNode;
import no.esito.jvine.model.TreeNodeImpl;
import no.esito.jvine.validation.ValidationManager;
import no.esito.jvine.validation.ValidationManagerFactory;
import no.esito.jvine.view.MessageUtil;
import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.action.CheckType;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.Interceptor;
import no.g9.client.core.controller.Interceptor.DIRECTIVE;
import no.g9.client.core.controller.RoleState;
import no.g9.client.core.validator.ValidateContext;
import no.g9.client.core.validator.ValidationPolicy.Policy;
import no.g9.client.core.validator.ValidationResult;
import no.g9.client.core.view.ViewModel;
import no.g9.os.AttributeConstant;
import no.g9.os.Key;
import no.g9.os.KeyTool;
import no.g9.os.OSRole;
import no.g9.os.RelationCardinality;
import no.g9.os.RelationType;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.TypeTool;
import no.g9.support.Visitor;

/**
 * Representation of a node in the object selection.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 *
 * @param <T>
 *            the domain class of the object selection node.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public final class OSNode<T> implements DisplayableOSRole<T> {

    /** The logger */
    private static final Logger log = Logger.getLogger(OSNode.class);

    /** The dialog controller of this object selection node */
    private final DialogController dialogController;

    /** The node used to keep references to the role instances */
    protected TreeNode<T> treeNode;

    /**
     * Delegate to object selection role part of this displayable.
     */
    private OSRole<T> delegate;

    /**
     * The state of the node.
     */
    private RoleState state = RoleState.CLEARED;

    private boolean stateOverride = false;

    private T lastObtained;

    private Map<CheckType, Interceptor> interceptors =
            new EnumMap<CheckType, Interceptor>(CheckType.class);

    private final MessageUtil messageUtil = ServiceLoader
	.getService(MessageUtil.class);

    /**
     * Constructs a new object selection node.
     *
     * @param parent
     *            the parent of this node.
     * @param dialogController
     *            the dialog controller of this node.
     * @param osRole
     *            The object selection role that this os node delegates to.
     */
    public OSNode(OSNode<?> parent, DialogController dialogController,
            OSRole<T> osRole) {
        this.dialogController = dialogController;
        this.delegate = osRole;

        if (log.isDebugEnabled()) {
            log.debug("Creating tree node " + osRole.getRoleConstant()
                    + " parent: " + parent);
        }

        TreeNode<?> parentTreeNode;
        if (delegate.getRelationCardinality() == RelationCardinality.ROOT) {
            if (log.isTraceEnabled()) {
                log.trace(this + " getting the tree node sentinel.");
            }
            parentTreeNode =
                    JVineController.getInstance(dialogController).getSentinel();

        } else if (parent != null) {
            if (log.isTraceEnabled()) {
                log.trace(this + " setting up tree node impl.");
            }
            parentTreeNode = parent.treeNode;
        } else {
            throw new NullPointerException("Parent cannot be null");
        }
        treeNode = new TreeNodeImpl<T>(parentTreeNode, this);
        parentTreeNode.addChild(treeNode);
        delegate = osRole;

    }

    /**
     * Returns a collection of this node's changed attributes.
     *
     * @return the attributes that are changed in the view.
     */
    Collection<AttributeConstant> getChangedAttributes() {
        return dialogController.getChangedAttributes(getRoleConstant());
    }

    /**
     * Returns a collection of the changed attributes that are not part of any
     * key.
     *
     * @return the collection of changed no-key attributes.
     */
    Collection<AttributeConstant> getChangedNoKeyAttributes() {
        Collection<AttributeConstant> allChanged = getChangedAttributes();
        Collection<AttributeConstant> keyAttributes =
                new HashSet<AttributeConstant>();
        for (Key key : getKeys()) {
            keyAttributes.addAll(Arrays.asList(key.getAttributes()));
        }
        allChanged.removeAll(keyAttributes);
        return allChanged;
    }

    @Override
    public RelationType getRelationType() {
        return delegate.getRelationType();
    }

    @Override
    public RelationCardinality getRelationCardinality() {
        return delegate.getRelationCardinality();
    }

    @Override
    public TreeNode<T> getTreeNode() {
        return treeNode;
    }

    @Override
    public Collection<OSNode<?>> setCurrentInstance(Object instance) {
        return getTreeNode().setCurrentInstance(castToType(instance));
    }

    /**
     * Set the current instances for this role. This is only supported if the
     * current role is a root role.
     *
     * @param instances
     *            the collection of (root) instances.
     * @return the collection of changed nodes.
     */
    public Collection<OSNode<?>> setCurrentInstances(Collection<?> instances) {
        return getTreeNode().setCurrentRootInstances((Collection<T>) instances);
    }

    @Override
    public Collection<T> getAllInstances() {
        return getTreeNode().getInstances();
    }

    /**
     * Check if the specified object is the current instance in the tree model.
     *
     * @param obj
     *            the object to check.
     * @return <code>true</code> if the specified object is the current
     *         instance.
     */
    public boolean isCurrent(Object obj) {
        return obj != null && obj.equals(treeNode.getCurrentInstance());
    }

    @Override
    public FieldData getFieldData() {
        return JVineController.getInstance(dialogController).getFieldData(
                getRoleConstant());
    }

    @Override
    public AttributeConstant[] getAttributeConstants() {
        return delegate.getAttributeConstants();
    }

    @Override
    public Object getRelation(Object domainObject, RoleConstant role) {
        return delegate.getRelation(domainObject, role);
    }

    @Override
    public void setMainKey(Key mainKey) {
        delegate.setMainKey(mainKey);
    }

    @Override
    public void setRelation(Object domainInstance, Object relation,
            RoleConstant role) {
        delegate.setRelation(domainInstance, relation, role);
    }

    @Override
    public void setValue(Object domainObject, AttributeConstant attribute,
            Object value) {
        delegate.setValue(domainObject, attribute, value);
    }

    @Override
    public T getCurrentInstance() {
        return getTreeNode().getCurrentInstance();
    }

    @Override
    public void updateRelation(Object domainInstance, Object relation,
            RoleConstant role) {
        delegate.updateRelation(domainInstance, relation, role);
    }

    /**
     * Builds a list of current objects for this node and all many-related child
     * nodes.
     *
     * @return a list of current role objects
     */
    public List<CurrentRoleObject> getCurrentObjectList() {
        List<CurrentRoleObject> currentList =
                new LinkedList<CurrentRoleObject>();
        T current = getCurrentInstance();
        if (current != null && isMany()) {
            currentList.add(new CurrentRoleObject(getRoleConstant(), current));
        }

        if (current != null) {
            for (OSNode<?> child : getOSNodeChildren()) {
                currentList.addAll(child.getCurrentObjectList());
            }
        }

        return currentList;
    }

    private void setFieldValue(AttributeConstant attribute, Object value) {
        dialogController.setFieldValue(attribute, value);
    }

    private void connectToChildren(Object instance) {
        Collection<OSNode<?>> osNodeChildren = getOSNodeChildren();
        for (OSNode<?> child : osNodeChildren) {
            Object childInst = child.peek();
            if (child.isMany()) {
                Collection<Object> defaultCollection =
                        TypeTool.getDefaultCollection();
                defaultCollection.add(childInst);
                setRelation(instance, defaultCollection, child
                        .getRoleConstant());
            } else {
                setRelation(instance, childInst, child.getRoleConstant());
            }
            child.connectToChildren(childInst);
        }
    }

    @Override
    public Collection<OSNode<?>> setCurrent(FieldData data) {
        if (log.isDebugEnabled()) {
            log.debug(this + " setting current instance for " + data);
        }

        final T mockInstance = createInstanceFromFieldData(data);
        if (isNavigableToParent()) {
            Object parentObj = treeNode.getParent().getCurrentInstance();
            setRelation(mockInstance, parentObj, getParent().getRoleConstant());
        }

        // Set relation to children in order to get a complete key...
        connectToChildren(mockInstance);

        Collection<T> instances = treeNode.getInstances();
        T found = null;
        Iterator<T> instanceIterator = instances.iterator();

        // Try first using equals.
        while (instanceIterator.hasNext() && found == null) {
            T instance = instanceIterator.next();
            if (instance != null && instance.equals(mockInstance)) {
                found = instance;
            }
        }

        // Try match on keys if equals failed.
        if (found == null) {
            instanceIterator = instances.iterator();
        }

        while (instanceIterator.hasNext() && found == null) {
            T instance = instanceIterator.next();
            if (matchUniqueKeys(mockInstance, instance, true)) {
                found = instance;
            }
        }

        if (found == null) {
            if (log.isDebugEnabled()) {
                log.debug("Could not find instance, making a new one.");
            }
            found = createNewInstance(data);
        } else {
            if (log.isDebugEnabled()) {
                log.debug("Found instance. Using that as the current");
            }
        }

        return treeNode.setCurrentInstance(found);
    }

    private T createInstanceFromFieldData(FieldData data) {
        if (log.isTraceEnabled()) {
            log.trace("Creating new instance based on " + data);
        }
        T instance = createNewInstance();
        for (Entry<AttributeConstant, Object> entry : data.entries()) {
            AttributeConstant attributeConstant = entry.getKey();
            Object attributeValue = entry.getValue();
            if (attributeConstant.isForeignAttribute()) {
                String msg =
                        "Can't create instance - expected local attribute but got foreign "
                                + attributeConstant;
                throw new IllegalArgumentException(msg);
            }

            if (attributeConstant.getAttributeRole().equals(getRoleConstant())
                    || isRelated(attributeConstant)) {
                if (log.isTraceEnabled()) {
                    log.trace("Setting attribute " + attributeConstant);
                }
                setValue(instance, attributeConstant, attributeValue);
            } else {
                if (log.isTraceEnabled()) {
                    log.trace("Ignoring unknown attribute " + attributeConstant
                            + ". It is neither local or related.");
                }
            }

        }

        return instance;
    }

    /**
     * Clears all data fields belonging to this object selection except fields
     * belonging to the main key, and clears all children.
     *
     * @return the collection of nodes that needs updating from the view.
     */
    public Collection<OSNode<?>> clearKeepKeys() {
        if (log.isTraceEnabled()) {
            log.trace(this + " clear keep keys.");
        }

        Collection<OSNode<?>> changed = new ArrayList<OSNode<?>>();

        getTreeNode().removeCurrent();
        AttributeConstant[] allAttributes = getAttributeConstants();
        Set<AttributeConstant> keyAttribute = new HashSet<AttributeConstant>();

        if (getMainKey() != null) {
            for (AttributeConstant attributeConstant : getMainKey()
                    .getAttributes()) {
                keyAttribute.add(attributeConstant);
            }
        }

        for (AttributeConstant attributeConstant : allAttributes) {
            if (!keyAttribute.contains(attributeConstant)) {
                setFieldValue(attributeConstant, null);
            }
        }

        for (OSNode<?> child : getOSNodeChildren()) {
            changed.addAll(child.clear(true));
        }

        if (log.isTraceEnabled()) {
            log.trace(this + " cleared kept keys. Changed nodes: " + changed);
        }

        return changed;

    }

    @Override
    public Collection<OSNode<?>> clear(boolean intercept) {
        if (log.isTraceEnabled()) {
            log.trace(this + " clearing");
        }
        return getTreeNode().clear(intercept);

    }

    @Override
    public Collection<OSNode<?>> clearCurrent(boolean intercept) {
        if (log.isTraceEnabled()) {
            log.trace(this + " clearing current");
        }
        return getTreeNode().clearCurrent(intercept);
    }

    private boolean isChanged() {
        return dialogController.isChanged(getRoleConstant());
    }

    /**
     * Obtains the current displayed instance, using the currently displayed
     * attribute values.
     * <strong>Note:</strong> This will obtain the complete object graph, including nodes <em>above</em> this node in the object selection.
     * @param clearCurrent if true the instance is removed as current instance
     *
     * @return the current displayed instance.
     */
    public T obtain(boolean clearCurrent) {
        if (log.isDebugEnabled()) {
            log.debug(this + " obtaining.");
        }

        T self = obtainCurrent(true);

        if (self == null) {
            return self;
        }

        // obtain children
        Collection<OSNode<?>> children = getOSNodeChildren();
        for (OSNode<?> child : children) {
            if (!child.hasValue()) {
                continue;
            }
            Object obtainedChild = child.obtainWideAsChild();
            if (child.isMany()) {
                addToCollection(child, self, obtainedChild);
            } else {
                setRelation(self, obtainedChild, child.getRoleConstant());
            }
        }

        // obtain parent if part of key or if clearCurrent
        OSNode<?> parentOSNode = getParentOSNode();
        if (parentOSNode != null && (isNavigableToParent() || clearCurrent)) {
            Object obtainedParentWithKey = parentOSNode.obtainToGetKey(parentOSNode.isPartOfKey(this)?this:null);
            if (isNavigableToParent()) {
                setRelation(self, obtainedParentWithKey, getParentRoleConstant());
            }

            if (clearCurrent) {
                Object relationFromParentToThis = parentOSNode.getRelation(obtainedParentWithKey, getRoleConstant());
                if (relationFromParentToThis instanceof Collection) {
                    ((Collection) relationFromParentToThis).remove(self);
                } else {
                    parentOSNode.setRelation(obtainedParentWithKey, null, getRoleConstant());
                }
            }

        }

        JVineController.getInstance(dialogController)
        .checkValidationAndConvert(true);

        return self;

    }


    private void addToCollection(OSNode<?> child, Object domainObject, Object obtainedChild) {
        assert child.isMany();
        Collection childCollection = (Collection) getRelation(domainObject, child.getRoleConstant());
        if (childCollection == null) {
            childCollection = new HashSet();
            setRelation(domainObject, childCollection, child.getRoleConstant());
        }
        childCollection.add(obtainedChild);
    }


    /**
     * Obtain node as child. All children are obtained if a current instance is present.
     * @return the obtained child.
     */
    private T obtainWideAsChild() {
        if (!hasValue()) {
            return null;
        }
        log.debug(this + " obtaining wide as child");
        T obtainedCurrent = obtainCurrent(false);
        Collection<OSNode<?>> children = getOSNodeChildren();
        for (OSNode<?> child : children) {
                Object obtainedChild = child.obtainWideAsChild();
                log.debug(this + " setting relation to obtained " + child);
                if (child.isMany()) {
                    addToCollection(child, obtainedCurrent, obtainedChild);
                }
                else {
                	setRelation(obtainedCurrent, obtainedChild, child.getRoleConstant());
                }
        }

        return obtainedCurrent;
    }

    /**
     * Obtain this node in order to get a complete key for the invoking child  node.
     * <ul>
     * <li>This node is obtained.
     * <li>Up-related children are possibly obtained (if part of this node's key)
     * <li>Relation from this to parent is set
     * <li>If parent node is part of this node's key it is obtained.
     * <li>relation from this to child is set
     * </ul>
     * @return the obtained instance
     */
    private T obtainToGetKey(OSNode invokingChildNode) {
        T obtainedCurrent = obtainCurrent(true);

        log.debug(this + " obtained to get complete key of " + invokingChildNode +
                " or to enable removal of " + invokingChildNode);

        // Obtain uprelated children with key
        List<OSNode<?>> upRelatedChildren = getUpRelatedChildren();
        for (OSNode<?> upRelatedchildNode : upRelatedChildren) {

            // We don't want to obtain the invoking child (avoid recursive loop)
            if (upRelatedchildNode.equals(invokingChildNode)) {
                continue;
            }

            if (upRelatedchildNode.isPartOfKey(this)) {
                Object obtainedChildWithKey = upRelatedchildNode.obtainToGetKey(null);
                log.debug(this + " setting relation to obtained " + upRelatedchildNode +
                        " in order to ensure a complete key");
                setRelation(obtainedCurrent, obtainedChildWithKey, upRelatedchildNode.getRoleConstant());
            }
        }

        /* If invoked by a child - continue to test upwards for keys.
         * If invoked by a parent (invokingChildNode == null) then
         * parent must already be obtained and this node is an up-related.
         */
        if (invokingChildNode != null) {
            OSNode<?> parentNode = getParentOSNode();
            if (parentNode != null && parentNode.isPartOfKey(this)) {
                Object obtainedParentKey = parentNode.obtainToGetKey(this);
                setRelation(obtainedCurrent, obtainedParentKey, getParentRoleConstant());
            }
        }

        return obtainedCurrent;

    }

    /**
     * Test if this node is part of the requesting node's key.
     * @param node the node requesting the key info
     * @return <code>true</code> if this node is part of the other's key
     */
    private boolean isPartOfKey(OSNode node) {
        List<Key> keys = node.getKeys();
        for (Key key : keys) {
            AttributeConstant[] attributes = key.getAttributes();
            for (AttributeConstant attributeConstant : attributes) {
                if (attributeConstant.isForeignAttribute()) {
                    RoleConstant foreignAttributeRole = attributeConstant.getForeignAttributeRole();
                    if (getRoleConstant().equals(foreignAttributeRole)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }


    /**
     * Perform check save on this node.
     *
     * @param action
     *            passed on to ValidateContext
     *
     * @return true if check save passed, otherwise false.
     */
    public CheckResult checkSave(ActionType action) {
        if (log.isTraceEnabled()) {
            log.trace("Check save " + this);
        }

        CheckResult csr = new CheckResult();

        DIRECTIVE directive =
                checkNode(CheckType.SAVE, getRoleConstant(), action, csr);

        boolean checkOK = directive == DIRECTIVE.CHANGED ? false : true;

        if (log.isDebugEnabled()) {
            log.debug("Check save " + this + (checkOK ? " passed" : " failed")
                    + ".");
        }
        csr.setCheckResult(Boolean.valueOf(checkOK));

        return csr;
    }

    private DIRECTIVE checkNode(CheckType checkType, RoleConstant target,
            ActionType action, CheckResult csr) {
        DIRECTIVE directive =
                checkNodeAsParent(checkType, target, this, action, csr);

        switch (directive) {
        case CHANGED:
        case UNCHANGED:
            return directive;
        default:
            break;
        }

        directive =
                checkSelfAndChildren(checkType, target, null, false, false,
                        action, csr);

        return directive;
    }

    private DIRECTIVE checkNodeAsParent(CheckType checkType,
            RoleConstant target, OSNode<?> ignoredChild, ActionType action,
            CheckResult csr) {
        DIRECTIVE directive = DIRECTIVE.DEFAULT;

        if (getParent() != null) {
            OSNode<?> parentNode =
                    dialogController.getOSRole(getParent().getRoleConstant());
            directive =
                    parentNode.checkNodeAsParent(checkType, target, this,
                            action, csr);
        }

        switch (directive) {
        case CHANGED:
        case UNCHANGED:
            return directive;
        default:
            break;
        }

        directive =
                checkSelfAndChildren(checkType, target, ignoredChild, true,
                        true, action, csr);

        return directive;

    }

    private DIRECTIVE checkSelfAndChildren(CheckType checkType,
            RoleConstant target, OSNode<?> ignoreChild, boolean asUprelated,
            boolean selfCheck, ActionType action, CheckResult csr) {
        DIRECTIVE directive = DIRECTIVE.DEFAULT;

        if (selfCheck) {
            if (log.isTraceEnabled()) {
                log.trace("Invoking check " + checkType + " interceptor on "
                        + this);
            }
            Interceptor interceptor = getInterceptor(checkType);
            if (interceptor != null) {
                directive = interceptor.intercept(target);
            }
        }

        switch (directive) {
        case CHANGED:
        case UNCHANGED:
            if (log.isTraceEnabled()) {
                log.trace("Interceptor stopped iteration.");
            }
            return directive;
        default:
            break;
        }

        if (directive == DIRECTIVE.DEFAULT && selfCheck) {

            Map<ValidationResult, ValidateContext> validateResult =
                    validate(action, target, Policy.ON_SAVE);
            for (Entry<ValidationResult, ValidateContext> entry : validateResult
                    .entrySet()) {
                ValidationResult key = entry.getKey();
                if (!key.succeeded()) {
                    csr.addValidationResult(key, entry.getValue());
                }
            }

            if (hasConversionErrors()) {
                Map<DialogObjectConstant, Collection<?>> ccm =
                        getConversionContextMessages();
                csr.addConversionError(ccm);
            }
        }
        if (selfCheck) {
            if (log.isTraceEnabled()) {
                log.trace("Check " + checkType + " of " + this
                        + " done. Continuing with children.");
            }
        }

        Collection<OSNode<?>> children;

        if (asUprelated) {
            children = getUpRelatedChildren();
        } else {
            children = getOSNodeChildren();
        }

        if (ignoreChild != null) {
            children.remove(ignoreChild);
        }

        for (OSNode<?> node : children) {
            DIRECTIVE childDirective =
                    node.checkSelfAndChildren(checkType, target, null,
                            asUprelated, true, action, csr);
            if (childDirective == DIRECTIVE.CHANGED) {
                if (log.isTraceEnabled()) {
                    log.trace("Check " + checkType + " of " + node + " is "
                            + childDirective);
                }
                directive = childDirective;
                break;
            }
        }

        return directive;

    }

    /**
     * Run all validators with the specified policy for this role.
     *
     * @param actionType
     *            action type to set in ValidateContext
     * @param policy
     *            policy policy to use when finding validators
     * @param actionTarget
     *            action target as role constant
     * @return map of all validation results (both succeeded and failed)
     */
    public Map<ValidationResult, ValidateContext> validate(
            ActionType actionType, RoleConstant actionTarget, Policy policy) {
        ValidationManager valMngr =
                ValidationManagerFactory.create(actionType, dialogController);
        Map<ValidationResult, ValidateContext> results =
                new LinkedHashMap<ValidationResult, ValidateContext>();
        for (AttributeConstant attribute : getAttributeConstants()) {
            results.putAll(valMngr.validate(attribute, actionTarget, policy));
        }
        return results;
    }

    /**
     * Perform check close on this node.
     *
     * @return true if this node and all child nodes are unchanged.
     */
    public boolean checkClose() {
        return checkChanged(CheckType.CLOSE, null).getCheckResult();
    }

    /**
     * Check if this node, or any of its child nodes are changed. A node is
     * considered <em>changed</em> if it's either {@link RoleState#DIRTY DIRTY}
     * or {@link RoleState#EDITED EDITED}.
     *
     * @param checkType
     *            the check type.
     * @param target
     *            the action target.
     * @return true if this node and all child nodes are unchanged.
     */
    public CheckResult checkChanged(CheckType checkType, final RoleConstant target) {

        if (log.isTraceEnabled()) {
            log.trace("Check " + checkType + ": " + this);
        }

        boolean checkOK = true;
        Interceptor.DIRECTIVE directive = Interceptor.DIRECTIVE.DEFAULT;

        Interceptor interceptor = getInterceptor(checkType);

        if (interceptor != null) {
            directive = interceptor.intercept(target);
            if (log.isTraceEnabled()) {
                log.trace("Check " + checkType + " directive: " + directive);
            }
        }
        CheckResult cr = new CheckResult();
        switch (directive) {
        case CHANGED:
            cr.setCheckResult(Boolean.FALSE);
            return cr; // stop iteration
        case UNCHANGED:
            cr.setCheckResult(Boolean.TRUE);
            return cr; // stop iteration
        case DEFAULT:
            checkOK = isCleanOrCleared(); // possibly continue iteration
            Map<DialogObjectConstant, Collection<?>> convMessages =
                    getConversionContextMessages();
            if (!convMessages.isEmpty()) {
                cr.addConversionError(convMessages);
            }
            break;
        case CONTINUE:
            checkOK = true; // continue iteration
            break;
        }

        cr.setCheckResult(checkOK);

        if (checkOK) {
            CheckResult child = checkChildren(checkType, target);
            cr.addConversionError(child.getConversionError());
            if (!child.getCheckResult()) {
                cr.setCheckResult(child.getCheckResult());
            }
        }
        return cr;
    }

    private Interceptor getInterceptor(CheckType checkType) {
        return interceptors.get(checkType);
    }

    /**
     * Adds the specified interceptor to this node.
     *
     * @param checkType
     *            the check the interceptor intercepts.
     * @param interceptor
     *            the interceptor.
     */
    public void addInterceptor(CheckType checkType, Interceptor interceptor) {
        interceptors.put(checkType, interceptor);
    }

    /**
     * Check the children of this node.
     *
     * @param checkType
     *            the check being performed
     * @param target
     *            the target of the ongoing action
     * @return the combined result of checking all the children of this node.
     */
    CheckResult checkChildren(CheckType checkType, final RoleConstant target) {
        if (log.isTraceEnabled()) {
            log.trace("Cheking children of " + this);
        }
        CheckResult cr = new CheckResult();
        cr.setCheckResult(Boolean.TRUE);

        Iterator<OSNode<?>> childIterator = getOSNodeChildren().iterator();
        while (childIterator.hasNext() && cr.getCheckResult()) {
            OSNode<?> childNode = childIterator.next();
            CheckResult childResult = childNode.checkChanged(checkType, target);
            cr.addConversionError(childResult.getConversionError());
            if (!childResult.getCheckResult()) {
                cr.setCheckResult(childResult.getCheckResult());
            }
        }
        return cr;
    }

    private boolean isCleanOrCleared() {
        RoleState roleState = getState();
        boolean isUnchanged =
                roleState == RoleState.CLEAN || roleState == RoleState.CLEARED;
        if (log.isTraceEnabled()) {
            String msg = "Check change - calculated state of " + this + " is: ";
            if (isUnchanged) {
                msg += "un";
            }
            msg += "changed.";
            log.trace(msg);
        }
        return isUnchanged;
    }

    /**
     * Perform check find on this node.
     *
     * @return true if find action can continue.
     */
    public CheckResult checkFind() {
        if (log.isTraceEnabled()) {
            log.trace("Check find: " + this);
        }

        boolean checkOK = true;
        CheckType checkType = CheckType.FIND;
        Interceptor.DIRECTIVE directive = Interceptor.DIRECTIVE.DEFAULT;

        Interceptor interceptor = getInterceptor(checkType);

        if (interceptor != null) {
            directive = interceptor.intercept(getRoleConstant());
            if (log.isTraceEnabled()) {
                log.trace("Check " + checkType + " directive: " + directive);
            }
        }
        CheckResult checkResult = new CheckResult();
        switch (directive) {
        case CHANGED:
            checkResult.setCheckResult(Boolean.FALSE);
            return checkResult; // stop iteration
        case UNCHANGED:
            checkResult.setCheckResult(Boolean.TRUE);
            return checkResult; // stop iteration
        case DEFAULT:
            checkOK = getChangedNoKeyAttributes().isEmpty();
            Map<DialogObjectConstant, Collection<?>> convMsg =
                    getConversionContextMessages();
            checkResult.addConversionError(convMsg);
            break;
        case CONTINUE:
            checkOK = true; // continue iteration
            break;
        }

        if (log.isTraceEnabled()) {
            String msg = "Check find " + this + " is ";
            if (checkOK) {
                msg += "unchanged. Continuing to check children.";
            } else {
                msg += "changed.";
            }
            log.trace(msg);
        }


        checkOK = checkOK && checkChildren(CheckType.FIND, getRoleConstant()).getCheckResult();
        checkResult.setCheckResult(checkOK);
        return checkResult;
    }

    /**
     * Test if this role contains any conversion errors.
     * @return <code>true</code> if this role contains any conversion errors.
     */
    public boolean hasConversionErrors() {
        JVineController jCtrl = JVineController.getInstance(dialogController);
        if (messageUtil.hasContextMessages(jCtrl)) {
            Collection<DialogObjectConstant> dialogObjects = getDialogObjects();
            for (DialogObjectConstant dialogObjectConstant : dialogObjects) {
                if (messageUtil.hasContextMessages(jCtrl, dialogObjectConstant)) {
                    return true;
                }
            }
        }
        return false;
    }

    private Collection<DialogObjectConstant> getDialogObjects() {
        List<DialogObjectConstant> fields = new ArrayList<DialogObjectConstant>();
        AttributeConstant[] attributeConstants = getAttributeConstants();
        ViewModel viewModel = dialogController.getDialogView().getViewModel();
        for (AttributeConstant attribute : attributeConstants) {
            Collection<DialogObjectConstant> dialogObjects = viewModel.getAttributeFields(attribute);
            fields.addAll(dialogObjects);
        }
        return fields;
    }

    /**
     * Get a collection of conversion error messages for this role.
     * @return collection of conversion error messages
     */
    public Map<DialogObjectConstant, Collection<?>> getConversionContextMessages()  {
        JVineController jCtrl = JVineController.getInstance(dialogController);
        Map <DialogObjectConstant, Collection<?>> cm = new HashMap<DialogObjectConstant, Collection<?>>();
        if (messageUtil.hasContextMessages(jCtrl)) {
            Collection<DialogObjectConstant> dialogObjects = getDialogObjects();
            for (DialogObjectConstant dialogObjectConstant : dialogObjects) {
                List contextMessages = messageUtil.getContextMessages(jCtrl, dialogObjectConstant);
                if (contextMessages != null && !contextMessages.isEmpty()) {
                    cm.put(dialogObjectConstant, contextMessages);
                }
            }
        }
        return cm;
    }

    private RoleConstant getParentRoleConstant() {
        return getParent() != null ? getParent().getRoleConstant() : null;
    }

    /**
     * Obtains the current node.
     *
     * @param conditional
     *            if <code>true</code> the node will only be obtained if it has
     *            a value.
     * @return the obtained object.
     */
    private T obtainCurrent(boolean conditional) {
        if (log.isTraceEnabled()) {
            log.trace("Obtaining current " + this);
        }
        Object[] create = conditionalCreateInstance(conditional);
        T instance = (T) create[0];

        if (instance != null) {
            if (!isCleanOrCleared()) {
                setFields(instance, (Boolean) create[1]);
            }
        }

        lastObtained = instance;

        return instance;
    }

    private void setFields(T instance, Boolean newObject) {
        if (newObject.booleanValue()) {
            setAllFields(instance);
        } else {
            setChangedFields(instance);
        }

    }

    private void setAllFields(T instance) {
        FieldData fieldData = getFieldData();
        for (Entry<AttributeConstant, Object> entry : fieldData.entries()) {
            logSetAttribute(entry.getKey(), entry.getValue());
            setValue(instance, entry.getKey(), entry.getValue());
        }
    }

    private void logSetAttribute(AttributeConstant attribute, Object value) {
        if (log.isTraceEnabled()) {
            String msg = "Setting value for field : " + attribute;
            msg += ", value: " + value;
            if (value != null) {
                msg += ", type " + value.getClass();
            }
            log.trace(msg);
        }
    }

    private void setChangedFields(T instance) {
        FieldData fieldData = getFieldData();
        Collection<AttributeConstant> changedAttributes =
                getChangedAttributes();
        for (AttributeConstant changedField : changedAttributes) {
            Object value = fieldData.getFieldValue(changedField);
            logSetAttribute(changedField, value);
            setValue(instance, changedField, value);
        }

    }

    /**
     * Creates a new instance, sets field data and returns.
     *
     * @return the new instance with containing values from the dialog.
     */
    public T peek() {
        T obj = createNewInstance();
        setAllFields(obj);
        return obj;
    }

    /**
     * Conditionally creates an instance. If the domain instance exists, it is
     * returned, else a new one is created if the conditional is
     * <code>true</code>.
     *
     * @param conditional
     *            if <code>true</code> a new instance is created, otherwise not.
     * @return a domain instance or <code>null</code>
     */
    private Object[] conditionalCreateInstance(boolean conditional) {
        Object[] retVals = new Object[2];
        retVals[0] = getCurrentInstance();
        retVals[1] = Boolean.FALSE;

        if (retVals[0] == null) {
            boolean createNew = false;
            if (log.isTraceEnabled()) {
                log.trace(this + " has no current instance.");
            }
            if (conditional) {
                if (log.isTraceEnabled()) {
                    log.trace(this + " Checking condition...");
                }
                createNew = hasValue();
                if (log.isTraceEnabled()) {
                    if (createNew) {
                        log.trace(this + " has value - creating new instance.");
                    } else {
                        log.trace(this + " has no value - no new instance "
                                + "will be created.");
                    }
                }
            } else {
                if (log.isTraceEnabled()) {
                    log.trace(this + " unconditionally creating new instance.");
                }
                createNew = true;
            }

            if (createNew) {
                retVals[0] = createNewInstance();
                retVals[1] = Boolean.TRUE;
                setCurrentInstance(retVals[0]);

            } else if (log.isTraceEnabled()) {
                log.trace(this + " has no value. No instance is created.");
            }
        }

        return retVals;
    }

    /**
     * Creates a new domain instance based on the specified field values.
     *
     * @param data
     *            the field values
     * @return a new instance where attributes have the values defined in
     *         <code>data</code>.
     */
    public T createNewInstance(FieldData data) {
        T instance = createNewInstance();
        for (Entry<AttributeConstant, Object> entries : data.entries()) {
            setValue(instance, entries.getKey(), entries.getValue());
        }
        return instance;
    }

    /**
     * Compares the unique keys found in the instance field data (populated by the view) and the instance. If a unique key matches
     * this method return <code>true</code>.
     *
     * @param instanceToSearchFor the instance field data
     * @param possibleMatch the instance
     * @param failOnMissingKey true if an exception should be thrown when no unique key is found
     * @return true if the data and the instance has at least one unique key in common.
     */
    public boolean matchUniqueKeys(T instanceToSearchFor, T possibleMatch, boolean failOnMissingKey) {
        if (log.isTraceEnabled()) {
            log.trace(this + " matching unique keys.");
        }

        Key uniqueKey = KeyTool.getCompleteUniqueKey(instanceToSearchFor, this);

        if (uniqueKey == null) {
            if (failOnMissingKey) {
                String msg= "Missing unique key for " + getRoleConstant()
                        + ". Can't find a complete key in instance field data.";
                throw new RuntimeException(msg);
            }
            return false;
        }

        if (log.isDebugEnabled()) {
            log.debug("Using " + uniqueKey + " as the unique key");
        }

        return equalsUsingKey(uniqueKey, instanceToSearchFor, possibleMatch);
    }

    @Override
    public void addChild(OSRole<?> child) {
        delegate.addChild(child);
    }

    @Override
    public T castToType(Object obj) {
        return delegate.castToType(obj);
    }

    @Override
    public T createNewInstance() {
        return delegate.createNewInstance();
    }

    @Override
    public Map<AttributeConstant, Object> getAttributeValues(Object type) {
        return delegate.getAttributeValues(type);
    }

    @Override
    public List<OSRole<?>> getChildren() {
        return delegate.getChildren();
    }

    /**
     * Gets the collection of children as object selection nodes. The collection
     * is created each time, so it is safe to modify the contents.
     *
     * @return a collection of children as object selection nodes.
     */
    public Collection<OSNode<?>> getOSNodeChildren() {
        List<OSNode<?>> childrenOS = new ArrayList<OSNode<?>>();
        List<OSRole<?>> children = getChildren();
        for (OSRole<?> role : children) {
            OSNode<?> child =
                    dialogController.getOSRole(role.getRoleConstant());
            childrenOS.add(child);
        }

        return childrenOS;

    }

    /**
     * Gets the list of up related children.
     *
     * @return all children that are up related.
     */
    public List<OSNode<?>> getUpRelatedChildren() {
        List<OSRole<?>> allChildren = getChildren();
        List<OSNode<?>> upRelated = new ArrayList<OSNode<?>>();
        for (OSRole<?> child : allChildren) {
            if (RelationType.UP_RELATED == child.getRelationType()) {
                OSNode<?> node =
                        dialogController.getOSRole(child.getRoleConstant());
                upRelated.add(node);
            }
        }
        return upRelated;
    }

    @Override
    public OSRole<?> getChild(RoleConstant childRole) {
        return delegate.getChild(childRole);
    }

    @Override
    public Class<T> getDomainClass() {
        return delegate.getDomainClass();
    }

    @Override
    public Key getMainKey() {
        return delegate.getMainKey();
    }

    @Override
    public OSRole<?> getParent() {
        return delegate.getParent();
    }

    /**
     * Returns the parent object selection node of this node, or null if this is
     * a root node.
     *
     * @return the parent object selection node of this node.
     */
    private OSNode<?> getParentOSNode() {
        if (getParent() != null) {
            OSNode<?> parent =
                    dialogController.getOSRole(getParentRoleConstant());
            return parent;
        }
        return null;
    }

    /**
     * Gets the root of this role (roots return self).
     *
     * @return the root role of this role.
     */
    @Override
    public OSRole<?> getRoot() {
        return delegate.getRoot();
    }

    @Override
    public RoleConstant getRoleConstant() {
        return delegate.getRoleConstant();
    }

    @Override
    public List<Key> getKeys() {
        return delegate.getKeys();
    }

    @Override
    public Object getValue(Object domainObject, AttributeConstant attribute) {
        return delegate.getValue(domainObject, attribute);
    }

    @Override
    public Map<AttributeConstant, Object> getValues(Object instance,
            AttributeConstant[] attributes) {
        return delegate.getValues(instance, attributes);
    }

    @Override
    public boolean isNavigableToParent() {
        return delegate.isNavigableToParent();
    }

    /**
     * Returns the cardinality of the role.
     *
     * @return The cardinality of the role
     */
    public RelationCardinality getCardinality() {
        return delegate.getRelationCardinality();
    }

    @Override
    public boolean equalsUsingKey(Key keyToUse, Object anObject,
            Object anOtherObject) {
        return delegate.equalsUsingKey(keyToUse, anObject, anOtherObject);
    }

    @Override
    public boolean isAncestorOf(RoleConstant possibleHeir) {
        return delegate.isAncestorOf(possibleHeir);
    }

    @Override
    public boolean isRelated(AttributeConstant attribute) {
        return delegate.isRelated(attribute);
    }

    @Override
    public boolean isUpRelated() {
        return delegate.isUpRelated();
    }

    @Override
    public boolean isParentMany() {
        return delegate.isParentMany();
    }
    
    @Override
    public boolean isPersistent() {
        return delegate.isPersistent();
    }

    /**
     * Gets the state of the node.
     *
     * @return the state
     */
    public final RoleState getState() {
        if (log.isTraceEnabled()) {
            log.trace(this + " getting state");
        }
        if (!stateOverride) {
            state = calculateState(state);
            if (log.isTraceEnabled()) {
                log.trace(this + " calculated state is: " + state);
            }
        } else if (log.isTraceEnabled()) {
            log.trace(this + " using state override: " + state);
        }
        return state;
    }

    /**
     * Overrides the state of the node.
     *
     * @param state
     *            the state to set
     */
    public final void setState(RoleState state) {
        if (log.isTraceEnabled()) {
            log.trace(this + " setting state to " + state);
        }
        stateOverride = true;
        this.state = state;
    }

    @Override
    public boolean hasValue() {
        boolean hasValue = getState() != RoleState.CLEARED;

        if (log.isTraceEnabled()) {
            log.trace(this + " internal state is "
                    + (hasValue ? "changed." : "unchanged."));
        }

        Iterator<OSNode<?>> childIterator = getOSNodeChildren().iterator();
        while (!hasValue && childIterator.hasNext()) {
            OSNode<?> child = childIterator.next();
            hasValue = hasValue || child.hasValue();
        }

        return hasValue;
    }

    /**
     * Resets state override so that the next call to getState will calculate
     * the state.
     */
    public final void resetState() {
        if (log.isTraceEnabled()) {
            log.trace(this + " resetting state override.");
        }
        stateOverride = false;
    }

    private RoleState calculateState(RoleState currentState) {
        boolean isChanged = isChanged();
        if (log.isTraceEnabled()) {
            log.trace(this + " calculating state. Currents state is "
                    + currentState + ". Node is "
                    + (isChanged ? " changed." : " unchanged."));
        }

        RoleState calculated = null;

        if (isChanged) {
            switch (currentState) {
            case CLEARED:
                calculated = RoleState.EDITED;
                break;
            case CLEAN:
                calculated = RoleState.DIRTY;
                break;
            default:
                calculated = currentState;
            }
        } else {
            switch (currentState) {
            case EDITED:
                calculated = RoleState.CLEARED;
                break;
            case DIRTY:
                calculated = RoleState.CLEAN;
                break;
            default:
                calculated = currentState;
            }
        }

        return calculated;
    }

    @Override
    public String toString() {
        return getRoleConstant().toString();
    }

    @Override
    public boolean isMany() {
        return delegate.isMany();
    }

    @Override
    public boolean isRoot() {
        return delegate.isRoot();
    }

    /**
     * Gets the last obtained instance.
     *
     * @return the lastObtained instance of this role.
     */
    public final T getLastObtained() {
        return lastObtained;
    }

    /*
     * (non-Javadoc)
     *
     * @see no.g9.os.OSRole#visitBranch(no.g9.support.Visitor)
     */
    @Override
    public void visitBranch(Visitor<OSRole<?>> visitor) {
        delegate.visitBranch(visitor);
    }

    @Override
    public void accept(Visitor<OSRole<?>> visitor) {
        delegate.accept(visitor);

    }

    @Override
    public AttributeConstant getAttributeConstant(String attributeName) {
        return delegate.getAttributeConstant(attributeName);
    }

}
