/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import no.esito.jvine.action.ActionHelper;
import no.esito.jvine.action.ActionMethod;
import no.esito.jvine.action.ActionMethodRunner;
import no.esito.jvine.action.SimpleActionExecutor;
import no.esito.jvine.action.UAMessageCallback;
import no.esito.jvine.communication.SystemMessageUtils;
import no.esito.jvine.model.TreeNode;
import no.esito.jvine.model.TreeNodeSentinel;
import no.esito.jvine.validation.ValidationMessageHelper;
import no.esito.jvine.view.AbstractApplicationView;
import no.esito.jvine.view.ViewModelImpl;
import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.action.ActionExecutor;
import no.g9.client.core.action.ActionFactory;
import no.g9.client.core.action.ActionTask;
import no.g9.client.core.action.EventContext;
import no.g9.client.core.action.G9Action;
import no.g9.client.core.action.RemoteServiceTarget;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.DialogSetupValue;
import no.g9.client.core.controller.DisplayException;
import no.g9.client.core.controller.ObtainException;
import no.g9.client.core.message.BlockingMessageCallback;
import no.g9.client.core.message.FieldMessageContext;
import no.g9.client.core.validator.ValidateContext;
import no.g9.client.core.validator.ValidationResult;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.Resource;
import no.g9.client.core.view.ViewModel;
import no.g9.exception.G9ClientException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageContext;
import no.g9.message.MessageDispatcher;
import no.g9.message.MessageSystem;
import no.g9.os.OSRole;
import no.g9.os.RelationCardinality;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.ObjectSelection;
import no.g9.support.action.ActionTarget;
import no.g9.support.convert.ConvertException;

/**
 * The jVine dialog controller.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public abstract class StateController {

    private static final Logger log = Logger.getLogger(StateController.class);

    private TreeNodeSentinel treeNodeSentinel;

    /** Flag used to control whether to invoke hooks or not. */
    private boolean invokeHooks = true;

    /** List of validationResult */
    private final List<ValidationResult> validationResult = new ArrayList<ValidationResult>();

    private final List<ConvertException> convertExceptions = new ArrayList<ConvertException>();

    /** The view observable wrapper */
    private final CtrlObservable observable = new CtrlObservable();

    /** Map of nodes */
    private final Map<RoleConstant, OSNode<?>> nodes = new HashMap<RoleConstant, OSNode<?>>();

    /** Map of root nodes */
    private final Map<RoleConstant, OSNode<?>> roots = new HashMap<RoleConstant, OSNode<?>>();

    /**
     * Maps the name of an event method to the runnable that invokes the event
     * method.
     */
    private final Map<String, ActionMethod> eventMethods = new HashMap<String, ActionMethod>();

    private ActionMethodRunner actionMethodRunner;

    /** The action factory of this dialog controller */
    private final ActionFactory actionFactory = new ActionFactory();

    /** The action executor */
    private final ActionExecutor executor = new SimpleActionExecutor();

    /** The caller of this dialog */
    private DialogController caller;

    /** The dialog instance number. */
    private int dialogInstanceNumber;


    /**
     * @param roleConstant the role constant
     * @return the field data
     * @see no.esito.jvine.controller.JVineController#getFieldData(no.g9.os.RoleConstant)
     */
    final FieldData getFieldData(RoleConstant roleConstant) {
        return getViewModel().getCurrentFieldData(roleConstant);
    }

    /**
     * @return the view model
     * @see no.esito.jvine.controller.JVineController#getViewModel()
     */
    public abstract ViewModel getViewModel();

    /**
     * Get the dialog object constant denoting this dialog.
     *
     * @return the dialogObjectConstant of this dialog.
     */
    public abstract DialogConstant getDialogConstant();


    private final String DIALOG_NAME;

    /** The key used to wrap the dialog constant for this dialog */
    private DialogKey dialogKey;

    /** The key used to wrap the dialog constant and instance number for this dialog */
    private DialogInstance dialogInstance;

    /**
     * Construct a new StateController
     *
     * @param dialogName the name of the dialog.
     */
    public StateController(String dialogName) {
        if (dialogName == null) {
            throw new IllegalArgumentException("Dialog name can not be null");
        }
        this.DIALOG_NAME = dialogName;
    }

    /**
     * Get the invokeHooks flag used to determine whether actions should invoke
     * registered hooks or not. If this flag is set to true, action hooks are
     * invoked, otherwise action hooks are ignored.
     *
     * @return <code>true</code> if action hooks should be invoked.
     */
    final boolean isInvokeHooksOn() {
        return invokeHooks;
    }

    /**
     * Set the invoke hooks flag used to determine whether actions running from
     * this controller should invoke registered hooks or not.
     *
     * @param invokeHooks the invoke hooks property.
     * @see #isInvokeHooksOn()
     */
    final void setInvokeHooks(boolean invokeHooks) {
        this.invokeHooks = invokeHooks;
    }

    /**
     * Gets the dialog key for this dialog.
     *
     * @return the dialog key.
     */
    final synchronized DialogKey getDialogKey() {
        if (dialogKey == null) {
            dialogKey = new DialogKey(getDialogConstant());
        }
        return dialogKey;
    }

    /**
     * Gets the dialog instance key for this dialog.
     *
     * @return the dialog instance key.
     */
    final synchronized DialogInstance getDialogInstance() {
        if (dialogInstance == null) {
            dialogInstance = new DialogInstanceKey(getDialogConstant(), getDialogInstanceNumber());
        }
        return dialogInstance;
    }

    /**
     * @param target the target
     * @param actionType the action type
     * @return the action task
     * @see no.esito.jvine.controller.JVineController#getObtainDialogTask(ActionTarget, boolean)
     */
    final ActionTask<ObjectSelection> getObtainDialogTask(ActionTarget target, ActionType actionType) {
        return ActionTaskFactory.createObtainDialogTask(target, (DialogController) this, actionType);
    }

    /**
     * Gets the dialog view class of this dialog.
     *
     * @param <T> The dialog view type
     * @return the dialog view.
     */
    @SuppressWarnings("unchecked")
    final synchronized <T extends DialogView> T getDialogViewInternal() {
        ApplicationController applicationController = getApplicationController();
        AbstractApplicationView applicationView = (AbstractApplicationView) applicationController.getApplicationView();
        return (T) applicationView.getDialogView(getDialogInstance());
    }

    /**
     * Get the application controller for this dialog
     *
     * @return the application controller
     */
    protected abstract ApplicationController getApplicationController();

    /**
     * @param target the target
     * @param actionType the action type for this display
     * @return the action task
     * @see no.esito.jvine.controller.JVineController#getDisplayDialogTask(ActionTarget,
     *      boolean)
     */
    final ActionTask<Void> getDisplayDialogTask(ActionTarget target, ActionType actionType) {
        return ActionTaskFactory.createDisplayDialogTask(target, actionType, (DialogController) this);
    }

    /**
     * @param role the role
     * @return the collection
     * @see no.esito.jvine.controller.JVineController#getOneRelatedChildren(no.g9.os.RoleConstant)
     */
    final Collection<OSNode<?>> getOneRelatedChildren(RoleConstant role) {
        Collection<OSNode<?>> oneRelatedChildren = new HashSet<OSNode<?>>();
        for (OSRole<?> child : getOSNode(role).getChildren()) {
            if (child.getRelationCardinality() == RelationCardinality.ONE) {
                oneRelatedChildren.add(getOSNode(child.getRoleConstant()));
            }
        }
        return oneRelatedChildren;
    }

    /**
     * Perform the specified action.
     *
     * @param action the action
     * @param event the event
     * @param target the target
     */
    @SuppressWarnings("unchecked")
	public final void performAction(ActionType action, EventContext event, ActionTarget target) {
        G9Action<?> g9Action = null;
        if (action == ActionType.PRINT) {
            g9Action = getAction(action, target, Resource.class,
                    ActionTaskFactory.createExportAction((DialogConstant) target, getApplicationController()));

        } else if (action == ActionType.EXPORTTOSPREADSHEET) {
        	g9Action = getAction(action, target, Resource.class,
        			ActionTaskFactory.createSpreadsheetAction((DialogConstant) target, getApplicationController()));

        } else if (target instanceof DialogConstant) {
            DialogSetupValue<?> setupValue=null;
            ActionTask<Void> dialogActionTask = null;

            if (event.getEvent() instanceof SystemMessage) {
                dialogActionTask = ActionTaskFactory.createDialogActionTaskFromExternal(action, (DialogConstant) target, (DialogController) this);

                SystemMessage message = (SystemMessage) event.getEvent();
                if( ((DialogConstant)target).getInternalName().equals(message.port) ){
                    // Meldinger rett til dialog.
                    try {
                        setupValue = SystemMessageUtils.extractActionSetupValue(message, getApplicationController());
                        setExternalActionActiveInstance(action, (DialogConstant) target, (DialogSetupValue<String>) setupValue);
                    } catch (Exception e) {
                        // Not to worry This is perfectly OK
                    }
                }
            }
            else {
                dialogActionTask = ActionTaskFactory.createDialogActionTask(action, (DialogConstant) target, (DialogController) this);
            }
            g9Action = getAction(action, target, Void.class, dialogActionTask);
            g9Action.setActionSetupValue(setupValue);
        } else if (ActionHelper.isGuiAction(action)) {
            g9Action = getAction(action, target, Void.class,
                    ActionTaskFactory.createGuiActionTask(action, target, (DialogController) this));

        } else if (ActionHelper.isGrapeAction(action)) {
            g9Action = getAction(action, target, Object.class,
                    ActionTaskFactory.createGrapeAction(
                            (DialogController) this, action));

        } else if (action == ActionType.CLEAR) {
            g9Action = getAction(action, target, Void.class,
                    ActionTaskFactory.createClearTask((Enum<?>) target,
                            (DialogController) this));

        } else if (action == ActionType.INVOKE) {
            RemoteServiceTarget rst = (RemoteServiceTarget) target;
            g9Action = getAction(rst);

        } else if (action == ActionType.ACTIVATE) {
            g9Action = getAction(action, target, ActionTarget.class, ActionTaskFactory.createActivateTask(target));

        }
        if (g9Action != null) {
            if (log.isDebugEnabled()) {
                log.debug("Executing " + g9Action);
            }
            g9Action.setEvent(event);
            executor.execute(g9Action);
        } else {
            String msg = "Cannot perform action. Unknown action type: " + action;
            throw new RuntimeException(msg);
        }
    }

    
    /**
     * Set the active instance for Close, Show and Hide actions if the received message
     * has an instance payload. If the given dialog instance is available, it is set as
     * the active instance. If not, the active instance is not changed.
     *
     * @param action the action type
     * @param dialog the target
     * @param setupValue the payload with an instance number is in the value of the setupValue object
     */
    private void setExternalActionActiveInstance(ActionType action, DialogConstant dialog, DialogSetupValue<String> setupValue) {
        if (setupValue != null) {
        	SystemMessage msg = new SystemMessage(setupValue.getSetupValue());
        	if (SystemMessage.ACTIVE_INSTANCE_PORT.equals(msg.port)) {
        		switch (action) {
        		case CLOSE:
        		case SHOW:
        		case HIDE:
        			int instanceNumber = Integer.parseInt(msg.payload);
        			DialogInstanceKey instance = new DialogInstanceKey(dialog, instanceNumber);
        			getApplicationController().setActiveDialogInstance(instance);
        			break;
        		default:
        			// Do nothing
        		    break;
        		}
        	}
        }
    }

    /**
     * Gets the event method runnable specified by <code>eventMethodName</code>.
     *
     * @param eventMethodName the name of the method
     * @return the runnable capable of invoking the event method.
     */
    private ActionMethod getEventMethod(String eventMethodName) {
        return eventMethods.get(eventMethodName);
    }

    /**
     * @see no.esito.jvine.controller.JVineController#closeDialogController()
     */
    final void closeDialogController() {
        for (OSNode<?> rootNode : getRoots().values()) {
            rootNode.clear(false);
        }
        actionMethodRunner.shutDown();
    }

    /**
     * @return the action method runner
     * @see no.esito.jvine.controller.JVineController#getActionMethodRunner()
     */
    final ActionMethodRunner getActionMethodRunner() {
        return actionMethodRunner;
    }

    /**
     * @return the message callback.
     * @see no.esito.jvine.controller.JVineController#getMessageCallback()
     */
    final BlockingMessageCallback getMessageCallback() {
        BlockingMessageCallback msgCallback = ServiceLoader
                .getService(BlockingMessageCallback.class);

        if (msgCallback instanceof UAMessageCallback) {
            UAMessageCallback uaMsgCb = (UAMessageCallback) msgCallback;
            uaMsgCb.setGuiActionQueue(actionMethodRunner.getActionQueue());
        }

        return msgCallback;
    }

    /**
     * @param <T> the type
     * @param actionType the action type
     * @param target the target
     * @param targetType the target type
     * @param actionTask the action task
     * @return the action
     *
     * @see no.esito.jvine.controller.JVineController#getAction(no.g9.support.ActionType,
     *      ActionTarget, java.lang.Class, no.g9.client.core.action.ActionTask)
     */
    final <T> G9Action<T> getAction(ActionType actionType, ActionTarget target,
            Class<T> targetType, ActionTask<T> actionTask) {

        return actionFactory.getAction(actionType, target, targetType,
                actionTask, (DialogController) this);
    }

    /**
     * Get the specified invoke action.
     *
     * @param target the remote service target.
     * @return the action
     */
    final G9Action<?> getAction(RemoteServiceTarget target) {
        return actionFactory.getInvokeAction(target, (DialogController) this);
    }

    /**
     * @param actionMethodRunner the action method runner.
     * @see no.esito.jvine.controller.JVineController#setActionMethodRunner(no.esito.jvine.action.ActionMethodRunner)
     */
    final void setActionMethodRunner(ActionMethodRunner actionMethodRunner) {
        if (log.isTraceEnabled()) {
            log.trace("Setting action method runner: "
                    + actionMethodRunner.getClass().getName());
        }
        this.actionMethodRunner = actionMethodRunner;
        actionMethodRunner.setThreadNamePrefix(DIALOG_NAME);
    }

    /**
     * @param event the event
     * @see no.esito.jvine.controller.JVineController#dispatch(no.g9.client.core.action.EventContext)
     */
    final void dispatch(EventContext event) {
        log.debug("Dispaching event: " + event);
        ActionMethod actionMethod = getEventMethod(event.getMethodName());
        if (actionMethod != null) {
            try {
                actionMethod.setEvent(event);
                actionMethodRunner.doMethod(actionMethod);
            } finally {
                actionMethod.setEvent(null);
            }
        } else {
            String msg = "No such action method: " + event.getMethodName();
            throw new IllegalArgumentException(msg);
        }
    }

    /**
     * @param view the dialog view
     * @see no.esito.jvine.controller.JVineController#registerView(no.g9.client.core.view.DialogView)
     */
    final synchronized void registerView(final DialogView view) {
        observable.registerView(view);
        if (log.isDebugEnabled()) {
            log.debug(this + " registering " + view + " as an observer.");
        }
    }

    /**
     * @param role the role
     * @param <T> the role type
     * @return the collection
     * @see no.esito.jvine.controller.JVineController#getAllInstances(no.esito.jvine.controller.OSNode)
     */
    final <T> Collection<T> getAllInstances(OSNode<T> role) {
        return role.getAllInstances();
    }

    /**
     * @param inst the instance
     * @param role the role
     * @return true or false...
     * @see no.esito.jvine.controller.JVineController#isCurrent(java.lang.Object,
     *      no.esito.jvine.controller.OSNode)
     */
    final boolean isCurrent(Object inst, OSNode<?> role) {
        return role.isCurrent(inst);
    }

    /**
     * @param node the node
     * @see no.esito.jvine.controller.JVineController#addOSNode(no.esito.jvine.controller.OSNode)
     */
    final void addOSNode(OSNode<?> node) {
        nodes.put(node.getRoleConstant(), node);
        if (node.getParent() == null) {
            getRoots().put(node.getRoleConstant(), node);
        }
    }

    /**
     * @return the tree node
     * @see no.esito.jvine.controller.JVineController#getSentinel()
     */
    final synchronized TreeNode<?> getSentinel() {
        if (treeNodeSentinel == null) {
            treeNodeSentinel = new TreeNodeSentinel();
        }

        return treeNodeSentinel;
    }

    /**
     * @param result the validation result.
     * @see no.esito.jvine.controller.JVineController#addValidationResult(no.g9.client.core.validator.ValidationResult)
     */
    final void addValidationResult(ValidationResult result) {
        this.validationResult.add(result);
    }

    /**
     * @param converterException the exception
     * @see no.esito.jvine.controller.JVineController#addConverterException(no.g9.support.convert.ConvertException)
     */
    final void addConverterException(ConvertException converterException) {
        this.convertExceptions.add(converterException);
    }

    /**
     * @param isObtain boolean
     * @throws G9ClientException if so...
     * @see no.esito.jvine.controller.JVineController#checkValidationAndConvert(boolean)
     */
    final void checkValidationAndConvert(boolean isObtain)
            throws G9ClientException {
        try {
            List<ValidationResult> failedValidations = new ArrayList<ValidationResult>();
            List<ConvertException> failedConverters = new ArrayList<ConvertException>();
            boolean failed = false;

            if (!validationResult.isEmpty()) {
                failedValidations.addAll(validationResult);
                failed = true;
            }

            if (!convertExceptions.isEmpty()) {
                failedConverters.addAll(convertExceptions);
                failed = true;
            }

            if (failed) {
                if (isObtain) {
                    throw new ObtainException(failedValidations,
                            failedConverters);
                }
                throw new DisplayException(failedValidations, failedConverters);
            }

        } finally {
            validationResult.clear();
            convertExceptions.clear();
        }
    }

    /**
     * @return the list
     * @see no.esito.jvine.controller.JVineController#getValidationResult()
     */
    final List<ValidationResult> getValidationResult() {
        return validationResult;
    }

    /**
     * @param role the role
     * @see no.esito.jvine.controller.JVineController#clearCurrent(no.g9.os.RoleConstant)
     */
    final void clearCurrent(RoleConstant role) {
        OSNode<?> currentRole = getOSNode(role);
        if (log.isDebugEnabled()) {
            log.debug("Clearing current for " + currentRole);
        }

        Collection<OSNode<?>> changed = currentRole.clearCurrent(false);

        reportToView(changed, null, currentRole);

    }

    /**
     * @param <T> The domain type of the object selection node.
     * @param role the role
     * @return the object selection node
     * @see no.esito.jvine.controller.JVineController#getOSNode(no.g9.os.RoleConstant)
     */
    @SuppressWarnings("unchecked")
    final <T> OSNode<T> getOSNode(RoleConstant role) {
        OSNode<?> node = nodes.get(role);
        if (node == null) {
            String msg = "No such role " + role;
            throw new IllegalArgumentException(msg);
        }
        return (OSNode<T>) node;
    }

    /**
     * @param changedNodes the changed nodes
     * @param includeNodes the include nodes
     * @param excludeNode the exclude nodes.
     * @see no.esito.jvine.controller.JVineController#reportToView(java.util.Collection,
     *      java.util.Collection, no.esito.jvine.controller.OSNode)
     */
    final void reportToView(Collection<OSNode<?>> changedNodes,
            Collection<OSNode<?>> includeNodes, OSNode<?> excludeNode) {
        observable.reportToView(changedNodes, includeNodes, excludeNode);
        checkValidationAndConvert(false);
    }

    /**
     * @return the os node array
     * @see no.esito.jvine.controller.JVineController#getRootNodes()
     */
    final OSNode<?>[] getRootNodes() {
        return getRoots().values().toArray(new OSNode<?>[getRoots().size()]);
    }

    /**
     * @return true or false
     * @see no.esito.jvine.controller.JVineController#checkClose()
     */
    final boolean checkClose() {
        boolean checkOK = true;
        for (OSNode<?> rootNode : getRootNodes()) {
            if (!rootNode.checkClose()) {
                checkOK = false;
                break;
            }
        }

        return checkOK;
    }

    /**
     * @return a map
     * @see no.esito.jvine.controller.JVineController#getRoots()
     */
    final Map<RoleConstant, OSNode<?>> getRoots() {
        return roots;
    }

    /**
     * @param instanceFieldData the instance field data.
     * @see no.esito.jvine.controller.JVineController#setCurrent(no.esito.jvine.controller.FieldData)
     */
    final void setCurrent(FieldData instanceFieldData) {

        OSNode<?> currentRole = getOSNode(instanceFieldData.getRole());
        if (log.isDebugEnabled()) {
            log.debug("Setting current for " + currentRole);
        }

        Collection<OSNode<?>> changed = currentRole
                .setCurrent(instanceFieldData);
        reportToView(changed, null, currentRole);
    }

    /**
     * @param eventMethodName the name of the event method
     * @param eventMethodInvoker the method invoker
     * @see no.esito.jvine.controller.JVineController#registerEventMethod(String,
     *      ActionMethod)
     */
    final void registerEventMethod(String eventMethodName,
            ActionMethod eventMethodInvoker) {
        eventMethods.put(eventMethodName, eventMethodInvoker);
    }

    /**
     * @return the action factory
     * @see no.esito.jvine.controller.JVineController#getActionFactory()
     */
    final ActionFactory getActionFactory() {
        return actionFactory;
    }

    /**
     * @param checkType the check type
     * @param target the action target
     * @return the action task
     * @see no.esito.jvine.controller.JVineController#getCheckDialogTask(no.g9.support.ActionType,
     *      ActionTarget)
     */
    final ActionTask<Boolean> getCheckDialogTask(ActionType checkType, ActionTarget target) {
        return ActionTaskFactory.createCheckDialogTask(checkType, target,
                (DialogController) this);
    }

    /**
     * Push the caller to the caller stack.
     *
     * @param dialogController the caller that opened this dialog
     */
    void setCaller(DialogController dialogController) {
        caller = dialogController;
    }

    /**
     * Gets the 1-based dialog instance number.
     *
     * @return the dialog instance number
     */
    public int getDialogInstanceNumber() {
        return dialogInstanceNumber;
    }

    /**
     * Sets the dialog instance number.
     *
     * @param dialogInstanceNumber the new dialog instance number
     */
    public void setDialogInstanceNumber(int dialogInstanceNumber) {
        this.dialogInstanceNumber = dialogInstanceNumber;
    }

    /**
     * Get the caller of this dialog.
     *
     * @return the dialog controller that opened this dialog
     */
    DialogController getCaller() {
        return caller;
    }

    /**
     * Validate said field and interact if validation failed.
     *
     * @param listRow list row to validate (possibly null)
     * @param field field to validate
     * @param newValue the new value
     * @return <code>true</code> if validation passed, else <code>false</code>
     */
    public boolean validateField(ListRow listRow, DialogObjectConstant field,
            Object newValue) {
        ViewModelImpl model = (ViewModelImpl) getDialogViewInternal()
                .getViewModel();

        Map<ValidationResult, ValidateContext> result = model.validateField(
                listRow, field, newValue);

        if (ValidationMessageHelper.succeeded(result)) {
            return true;
        }

        actionMethodRunner.doMethod(new ValidationTask(field, result));
        return false;
    }

    /**
     * Get the message dispatcher for this dialog controller
     *
     * @return message dispatcher with a message context.
     */
    public abstract MessageDispatcher getMessageDispatcher();

    /**
     * Performs the interaction when validating. This is due to thread issues.
     */
    class ValidationTask implements Runnable {
        private final DialogObjectConstant field;
        private final Map<ValidationResult, ValidateContext> result;

        private ValidationTask(DialogObjectConstant field,
                Map<ValidationResult, ValidateContext> result) {
            this.field = field;
            this.result = result;
        }

        private MessageContext getFieldMessageContext(
                final DialogObjectConstant field) {
            return new FieldMessageContext() {

                @Override
                public DialogObjectConstant getField() {
                    return field;
                }

                @Override
                public String toString() {
                    return "FieldMessageContext [" + field + "]";
                }

            };
        }

		@Override
		public void run() {

			try {
				Object[] messageArgs = ValidationMessageHelper.getMessageArgs(result);
				Message message = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_VALIDATION_FAILED_MAIN, messageArgs);
				message.setContext(getFieldMessageContext(field));

				for (Map.Entry<ValidationResult, ValidateContext> entry : result.entrySet()) {
					ValidateContext validateContext = entry.getValue();

					if (result.size() > 1) {
						Message childMessage = MessageSystem.getMessageFactory().getMessage(entry.getKey().getMsgNumber(),
								entry.getKey().getMsgArgs());
						childMessage.setContext(validateContext);
						message.addMessage(childMessage);
					} else {
						message = MessageSystem.getMessageFactory().getMessage(entry.getKey().getMsgNumber(), entry.getKey().getMsgArgs());
						message.setContext(validateContext);
					}
				}
				MessageDispatcher dispatcher = getMessageDispatcher();
				dispatcher.dispatch(message);
			} finally {
				getActionMethodRunner().releaseGui();
			}
		}

    }

}
