/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.rpc;

import no.esito.jvine.controller.JVineController;
import no.esito.jvine.controller.OSNode;
import no.esito.log.Logger;
import no.g9.client.core.action.Parameter;
import no.g9.client.core.controller.DialogController;
import no.g9.os.AttributeConstant;
import no.g9.os.RoleConstant;

/**
 * Represents an actual parameter of a remote service
 *
 * @param <T>
 *            the parameter type.
 */
public class ActualParameter<T> implements Parameter<T> {

    private final static Logger log = Logger.getLogger(ActualParameter.class);

    private final String parameterName;
    private final ParameterType parameterType;
    private T parameterValue = null;
    private RoleConstant role;
    private AttributeConstant attribute;
    private final DialogController dialogController;
    private boolean isObtained;

    /**
     * Enumerates the possible actual parameter types.
     */
    public enum ParameterType {

        /** The parameter is a constant */
        CONSTANT,

        /** The parameter is an object selection role */
        ROLE,

        /** The parameter is an object selection role attribute */
        ATTRIBUTE,

        /** The parameter is not bounded */
        UNBOUND;
    }

    /**
     * Constructs a new actual parameter
     *
     * @param dialogController
     *            the dialog controller that contains the actual parameter
     *
     * @param parameterName
     *            the parameter name
     * @param parameterType
     *            the parameter type
     */
    public ActualParameter(DialogController dialogController,
            String parameterName, ParameterType parameterType) {
        this.dialogController = dialogController;
        this.parameterName = parameterName;
        this.parameterType = parameterType;
    }

    @Override
    public final String getParameterName() {
        return parameterName;
    }

    /**
     * Convenience method - test if the parameter type is
     * {@link ParameterType#CONSTANT}
     *
     * @return <code>true if this is a CONSTANT parameter.
     */
    public final boolean isConstant() {
        return getParameterType() == ParameterType.CONSTANT;
    }

    /**
     * Convenience method - test if the parameter type is
     * {@link ParameterType#ROLE}
     *
     * @return <code>true if this is a ROLE parameter.
     */
    public final boolean isRole() {
        return getParameterType() == ParameterType.ROLE;
    }

    /**
     * Convenience method - test if the parameter type is
     * {@link ParameterType#ATTRIBUTE}
     *
     * @return <code>true if this is a ATTRIBUTE parameter.
     */
    public final boolean isAttribute() {
        return getParameterType() == ParameterType.ATTRIBUTE;
    }

    /**
     * Get the type of this parameter.
     *
     * @return the parameter type.
     */
    public final ParameterType getParameterType() {
        return parameterType;
    }

    /**
     * Get the value of the parameter.
     *
     * @return the parameter value.
     */
    @Override
    public final T getParameterValue() {
        return parameterValue;
    }

    /**
     * Set the value of the parameter.
     * @param parameterValue
     *            the parameter value.
     */
    @Override
    public final void setParameterValue(T parameterValue) {
        this.parameterValue = parameterValue;
    }

    /**
     * Get the role of this parameter. If the parameter type is
     * {@link ParameterType#CONSTANT} the role is <code>null</code>. If the
     * parameter type is {@link ParameterType#ATTRIBUTE} the role of the
     * attribute is returned. If the parameter type is
     * {@link ParameterType#ROLE} the role is returned.
     *
     * @return the object selection role of this parameter or <code>null</code>
     *         if this is a constant.
     */
    public RoleConstant getRole() {
        return role;
    }

    /**
     * Set the role belonging to this parameter.
     *
     * @param role
     *            the role of this parameter.
     * @throws IllegalArgumentException
     *             if the attribute type is CONSTANT and role is not null.
     */
    public void setRole(RoleConstant role) {
        this.role = role;
        if (parameterType == ParameterType.CONSTANT && role != null) {
            throw new IllegalArgumentException("Constant parameter cannot "
                    + "have an associated role");
        }
    }

    /**
     * Get the attribute constant represented by this parameter
     *
     * @return if this is an attribute parameter, the attribute constant, else
     *         <code>null</code>
     */
    public AttributeConstant getAttribute() {
        return attribute;
    }

    /**
     * Set the attribute belonging to this parameter.
     *
     * @param attribute
     *            the attribute of this parameter.
     * @throws IllegalArgumentException
     *             if the parameter type is not ATTRIBUTE and the attribute is
     *             not null
     */
    public void setAttribute(AttributeConstant attribute) {
        this.attribute = attribute;
        if (parameterType != ParameterType.ATTRIBUTE && attribute == null) {
            throw new IllegalArgumentException("Attribute constant only "
                    + "allowed for actual parameters of type ATTRIBUTE");
        }
    }

    /**
     * Obtain the parameter value.
     * @return the obtained parameter value.
     */
    @SuppressWarnings("unchecked")
    public T obtainValue() {

        T value = null;
        switch (parameterType) {
        case UNBOUND:
            log.debug("Obtaining unbound - nothing to obtain");
            break;
        case CONSTANT:
            log.debug("Obtaining constant - nothing to obtain");
            value = getParameterValue();
            break;
        case ATTRIBUTE:
            log.debug("Obtaining attribute - getting field value");
            value = (T) dialogController.getFieldValue(attribute);
            break;
        case ROLE:
            log.debug("Obtaining role");
            JVineController jController = JVineController
                    .getInstance(dialogController);
            OSNode<T> osNode = jController.getOSNode(role);
            value = osNode.obtain(false);
            break;
        default:
            String msg = "Unknown actual parameter type " + parameterType;
            throw new RuntimeException(msg);
        }
        isObtained = true;
        return value;
    }

    /**
     * Test if this parameter has been obtained.
     * @return <code>true</code> if this parameter is obtained.
     */
    public boolean isObtained() {
        return isObtained;
    }

    @Override
    public String toString() {
        return "ActualParameter ["
                + (parameterName != null ? "parameterName=" + parameterName
                        + ", " : "")
                + (parameterType != null ? "parameterType=" + parameterType
                        + ", " : "")
                + (attribute != null ? "attribute=" + attribute + ", " : "")
                + (role != null ? "role=" + role + ", " : "")
                + (parameterValue != null ? "parameterValue=" + parameterValue
                        + ", " : "") + "isObtained=" + isObtained + "]";
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result
                + ((attribute == null) ? 0 : attribute.hashCode());
        result = prime * result
                + ((parameterName == null) ? 0 : parameterName.hashCode());
        result = prime * result
                + ((parameterType == null) ? 0 : parameterType.hashCode());
        result = prime * result
                + ((parameterValue == null) ? 0 : parameterValue.hashCode());
        result = prime * result + ((role == null) ? 0 : role.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActualParameter<?>)) {
            return false;
        }
        ActualParameter<?> other = (ActualParameter<?>) obj;
        if (attribute == null) {
            if (other.attribute != null) {
                return false;
            }
        } else if (!attribute.equals(other.attribute)) {
            return false;
        }
        if (parameterName == null) {
            if (other.parameterName != null) {
                return false;
            }
        } else if (!parameterName.equals(other.parameterName)) {
            return false;
        }
        if (parameterType == null) {
            if (other.parameterType != null) {
                return false;
            }
        } else if (!parameterType.equals(other.parameterType)) {
            return false;
        }
        if (parameterValue == null) {
            if (other.parameterValue != null) {
                return false;
            }
        } else if (!parameterValue.equals(other.parameterValue)) {
            return false;
        }
        if (role == null) {
            if (other.role != null) {
                return false;
            }
        } else if (!role.equals(other.role)) {
            return false;
        }
        return true;
    }
}
