/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.validation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import no.esito.jvine.view.ViewModelImpl;
import no.esito.util.BeanID;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.DialogObjectType;
import no.g9.client.core.controller.RoleState;
import no.g9.client.core.validator.FieldValidator;
import no.g9.client.core.validator.MandatoryValidator;
import no.g9.client.core.validator.ValidateContext;
import no.g9.client.core.validator.ValidationPolicy;
import no.g9.client.core.validator.ValidationPolicy.Policy;
import no.g9.client.core.validator.ValidationResult;
import no.g9.client.core.view.DialogView;
import no.g9.os.AttributeConstant;
import no.g9.os.OSRole;
import no.g9.os.RoleConstant;
import no.g9.service.G9Spring;
import no.g9.support.ActionType;

/**
 * Utility class. Used to perform validation. Instances of this class should be
 * created by the ValidationManagerFactory.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
@BeanID("validationManager")
public final class ValidationManager {

    private Policy defaultPolicy = Policy.ON_SAVE;

    private ActionType actionType;

    private DialogController dctrl;

    private ValidationManager() {
        super();
    }

    /**
     * Set the default validation policy. If a validator is not annotated with
     * the {@link ValidationPolicy} annotaion,
     * 
     * @param defaultPolicy
     *            the default validation policy
     */
    public void setDefaultPolicy(Policy defaultPolicy) {
        this.defaultPolicy = defaultPolicy;
    }

    /**
     * Get the default policy. The default policy is used when a validator is
     * not annotated with a policy.
     * 
     * @return default validation policy
     */
    public Policy getDefaultPolicy() {
        return defaultPolicy;
    }

    /**
     * Initiate the validation manager.
     * 
     * @param actionType
     *            action type of validation
     * @param dctrl
     *            dialog controller
     * @return the validation manager ready to use
     */
    ValidationManager initManager(ActionType actionType, DialogController dctrl) {
        ValidationManager manager = new ValidationManager();
        manager.defaultPolicy = defaultPolicy;
        manager.actionType = actionType;
        manager.dctrl = dctrl;
        return manager;
    }

    /**
     * Validate an attribute using the specified validation policy. All
     * validators (mandatory + modeled validators) are being run, and the result
     * is put in a map (from result to context).
     * 
     * @param attribute
     *            attribute to validate
     * @param policy
     *            used to discriminate which validators to run
     * @param actionTarget
     *            action target as role constant
     * @return map of validation results and corresponding context
     */
    public Map<ValidationResult, ValidateContext> validate(AttributeConstant attribute, RoleConstant actionTarget, Policy policy) {
        Map<ValidationResult, ValidateContext> resultMap = new LinkedHashMap<ValidationResult, ValidateContext>();
        Collection<DialogObjectConstant> attributeFields = dctrl.getViewModel().getAttributeFields(attribute);
        if (attributeFields != null) {
            DialogObjectConstant field = null;
            ValidateContext context = null;

            Iterator<DialogObjectConstant> iterator = attributeFields.iterator();
            while (iterator.hasNext()) {
                field = iterator.next();
                context = new ValidateContext(dctrl, field, actionType);
                DialogObjectConstant diaconst = context.getDialogObjectConstant();
                DialogObjectConstant parent = (DialogObjectConstant)diaconst.getParent();
                if (parent != null && !(parent.getType() == DialogObjectType.TableBlock)) {
                    break;
                }
            }

            List<FieldValidator> validators = getValidator(attribute, policy);
            if (validators != null && !validators.isEmpty()) {
                for (FieldValidator val : validators) {
                    if (doValidate(field, actionTarget)) {
                        Object value = getValue(field);
                        ValidationResult valResult = val.validate(value, context);
                        resultMap.put(valResult, context);
                    }
                }
            }
        }
        return passResultToController(policy, resultMap);
    }

    /**
     * Checks if validation should be executed.
     * 
     * @param field to validate
     * @param actionTarget .
     * @return true if validation, else false
     */
    boolean doValidate(DialogObjectConstant field, RoleConstant actionTarget) {
	    RoleState state = dctrl.getState(field.getAttribute().getAttributeRole());
	    if (state == RoleState.CLEAN || state == RoleState.CLEARED) {
	        if (field.getAttribute().getAttributeRole() == actionTarget) {
	            return true;
            }

	        OSRole<?> osRole= dctrl.getOSRole(field.getAttribute().getAttributeRole());
	        if (osRole.isUpRelated() && osRole.getParent() != null) {
	            RoleState parentState= dctrl.getState(osRole.getParent().getRoleConstant());
	            if (parentState != RoleState.CLEARED) {
	                return true;
	            }
	        }

	        List<OSRole<?>> children= dctrl.getOSRole(field.getAttribute().getAttributeRole()).getChildren();
	        // Check for downrelated
	        if (hasDirtyChildren(children) && isChildOfTarget(osRole, actionTarget)) {
	            return true;
	        }

	        return false;
	    }
        return true;
    }

    private boolean isChildOfTarget(OSRole<?> osRole, RoleConstant actionTarget) {
        if (osRole.getParent() == null) return false;

        if (osRole.getParent().getRoleConstant() == actionTarget) {
            return true;
        }

        return isChildOfTarget(osRole.getParent(), actionTarget);
    }

    private boolean hasDirtyChildren(List<OSRole<?>> children) {
        if (children == null || children.size() == 0) return false;
        for (OSRole<?> child : children) {
            RoleState state= dctrl.getState(child.getRoleConstant());
            if (state != RoleState.CLEARED) {
                return true;
            }
            if (hasDirtyChildren(child.getChildren())) {
                return true;
            }
        }
        return false;
    }

    private Map<ValidationResult, ValidateContext> passResultToController(
            Policy policy, Map<ValidationResult, ValidateContext> resultMap) {
        if (policy == Policy.ON_CHANGE) {
            resultMap = ValidationMessageHelper.getFailed(resultMap);
            if (!resultMap.isEmpty()) {
                resultMap = dctrl.failedOnChangeValidator(resultMap);
            }
        }
        return resultMap;
    }
    
    /**
     * Run validators for the specified field and value.
     * @param field the field to validate
     * @param value the (field) value to validate
     * @param policy validation policy
     * @param context validation context
     * @return map of validation results (both succeeded and failed)
     */
    public Map<ValidationResult, ValidateContext> validate(DialogObjectConstant field, Object value, Policy policy, ValidateContext context) {
        Map<ValidationResult, ValidateContext> validationResults =
                new HashMap<ValidationResult, ValidateContext>();
        List<FieldValidator> validator = getValidator(field.getAttribute(), policy);
        for (FieldValidator fieldValidator : validator) {
            validationResults.put(fieldValidator.validate(value, context), context);
        }
        
        return passResultToController(policy, validationResults);
    }
    
    private Object getValue(DialogObjectConstant field) {
        DialogView dialogView = dctrl.getDialogView();
        ViewModelImpl viewModel = (ViewModelImpl) dialogView.getViewModel();
        return viewModel.getField(field);
    }

    private FieldValidator getMandatoryValidator(AttributeConstant attribute) {
        DialogView dialogView = dctrl.getDialogView();
        Collection<DialogObjectConstant> attributeFields =
                dialogView.getViewModel().getAttributeFields(attribute);
        for (DialogObjectConstant dialogObjectConstant : attributeFields) {
            if (dialogView.isMandatory(dialogObjectConstant)) {
                return new MandatoryValidator();
            }
        }
        return null;
    }

    private List<FieldValidator> getValidator(AttributeConstant attribute,
            Policy policy) {
        List<FieldValidator> validators = new ArrayList<FieldValidator>();

        FieldValidator mandatoryValidator = getMandatoryValidator(attribute);
        if (confirmPolicy(mandatoryValidator, policy)) {
            validators.add(mandatoryValidator);
        }

        FieldValidator validator = null;
        if (attribute != null) {
            String validatorID = attribute.getValidatorId();
            if (validatorID != null) {
                validator = G9Spring.getBean(validatorID);
                if (confirmPolicy(validator, policy)) {
                    validators.add(validator);
                }
            }

        }
        return validators;
    }

    /**
     * Confirm that the actual policy of a validator is of desired type. 
     * @param validator the validator to test policy on
     * @param policy the desired policy type
     * @return <code>true</code> if the validator's policy is the same as the specified policy
     */
    public boolean confirmPolicy(FieldValidator validator, Policy policy) {
        if (validator == null) {
            return false;
        }
        if (policy == null || policy == Policy.ON_SAVE) {
            return true;
        }
        
        return getValidatorType(validator) == policy;
    }

    private ValidationPolicy.Policy getValidatorType(FieldValidator validator) {
        Policy type = getDefaultPolicy();
        if (validator != null) {
            ValidationPolicy annotation =
                    validator.getClass().getAnnotation(ValidationPolicy.class);
            if (annotation != null) {
                type = annotation.value();
            }
        }
        return type;
    }

}
