/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.view;


import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import no.esito.jvine.action.GuiTask;
import no.esito.jvine.action.ThreadManager;
import no.esito.jvine.action.UAMessageCallback;
import no.esito.jvine.communication.SystemMessagePipe;
import no.esito.jvine.controller.DialogInstanceKeyFromExternal;
import no.esito.jvine.controller.JVineAppController;
import no.esito.jvine.controller.JVineController;
import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.action.CheckType;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogConstant.WindowType;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.message.BlockingMessageCallback;
import no.g9.client.core.message.InteractionThreadPolicy;
import no.g9.client.core.message.JVineDispatcherContext;
import no.g9.client.core.message.MessageCallback;
import no.g9.client.core.view.ApplicationView;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ViewModel;
import no.g9.client.core.view.WindowEvent;
import no.g9.message.CRuntimeMsg;
import no.g9.message.DispatcherContext;
import no.g9.message.Message;
import no.g9.message.MessageDispatcher;
import no.g9.message.MessageReplyType;
import no.g9.message.MessageSystem;
import no.g9.os.RoleConstant;
import no.g9.os.UsertypeRoleConstant;
import no.g9.service.G9Spring;
import no.g9.support.ActionType;

import com.google.common.collect.Lists;

/**
 * Abstract superclass for the generated application view, which holds a
 * reference to the application controller.
 *
 * This class holds maps of the dialog views and models of the application.
 * It also has knowledge of the currently opened dialogs, and handles the
 * message callback.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public abstract class AbstractApplicationView implements ApplicationView, SystemMessagePipe {

    private static final String VIEW = "View";

    /**
     * Common logger for this and its subclasses.
     */
    protected static final Logger log = Logger.getLogger(DialogView.class);

    private ApplicationController applicationController;

    private final Map<DialogInstance, ViewModel> viewModels =
        new HashMap<DialogInstance, ViewModel>();

    private final Map<DialogInstance, AbstractDialogView> dialogViews =
        new HashMap<DialogInstance, AbstractDialogView>();

    private final List<DialogInstance> openDialogs = new ArrayList<DialogInstance>();

    private final List<DialogInstance> openDialogBoxes = new ArrayList<DialogInstance>();

    private final Map<DialogInstance, Boolean> shown = new HashMap<DialogInstance, Boolean>();

    private final Map<DialogInstance, String> dialogTitle = new HashMap<DialogInstance, String>();

    private Message message;

    private MessageCallback messageCallback;

    private final List<SystemMessage> outputMessages = new ArrayList<SystemMessage>();

    /**
     * Create a new application view for the given application controller.
     *
     * @param applicationController the application controller to reference.
     */
    public AbstractApplicationView(ApplicationController applicationController) {
        setApplicationController(applicationController);
        applicationController.setApplicationView(this);
    }

    @Override
	public ApplicationController getApplicationController() {
        return applicationController;
    }

    /**
     * Set the referenced application controller.
     *
     * @param applicationController the new application controller.
     */
    protected void setApplicationController(ApplicationController applicationController) {
        this.applicationController = applicationController;
    }

    @Override
    public final synchronized ViewModel getViewModel(DialogConstant dialog) {
        return getViewModel(getCurrentDialogInstance(dialog));
    }

    /**
     * Get the view model for the given dialog instance.
     * If the view model is not found, a new view model is created.
     *
     * @param dialog the dialog instance
     * @return the view model
     */
    public final synchronized ViewModel getViewModel(DialogInstance dialog) {
        ViewModel viewModel = viewModels.get(dialog);
        if (viewModel == null) {
            viewModel = new ViewModelImpl(dialog, this);
            viewModels.put(dialog, viewModel);
        }
        return viewModel;
    }

    @Override
    @SuppressWarnings("unchecked")
    public final synchronized <T extends DialogView> T getDialogView(DialogConstant dialog) {
        DialogInstance instance = getCurrentDialogInstance(dialog);
        return (T) getDialogView(instance);
    }

    /**
     * Get the dialog view for the given dialog instance.
     * If the view is not open, it will be created.
     * If the special "empty" dialog is requested, null is returned.
     *
     * @param <T> the DialogView subclass to return
     * @param instance the dialog instance for the view
     *
     * @return an existing dialog view for the dialog instance
     */
    @SuppressWarnings("unchecked")
    public final synchronized <T extends DialogView> T getDialogView(DialogInstance instance) {
        if (isNullOrEmptyInstance(instance)) {
            return null;
        }
        AbstractDialogView dialogView = dialogViews.get(instance);
        if (dialogView == null) {
            dialogView = createDialogViewInternal(instance);
        }
        return (T) dialogView;
    }

    private boolean isNullOrEmptyInstance(DialogInstance instance) {
        if (instance == null) {
            return true;
        }
        String dialogName = getDialogName(instance.getDialogConstant());
        if (dialogName.equals(getEmptyDialogName())) {
            return true;
        }
        return false;
    }

    /**
     * Check if there is an open view for the given dialog instance.
     *
     * @param instance the dialog instance for the view
     * @return true if a view is open, false otherwise
     */
    public final synchronized boolean hasOpenDialogView(DialogInstance instance) {
        if (isNullOrEmptyInstance(instance)) {
            return false;
        }
        return dialogViews.containsKey(instance);
    }

    /**
     * Create a new dialog view for the given dialog instance.
     * If the view is already open, it will be returned as is.
     * If the special "empty" dialog is requested, null is returned.
     *
     * @param <T> the DialogView subclass to return
     * @param instance the dialog instance for the view
     *
     * @return a new dialog view for the dialog instance
     */
    @SuppressWarnings("unchecked")
    public synchronized <T extends AbstractDialogView> T createDialogView(DialogInstance instance) {
        String dialogName = getDialogName(instance.getDialogConstant());
        if (dialogName.equals(getEmptyDialogName())) {
            return null;
        }
        AbstractDialogView dialogView = dialogViews.get(instance);
        if (dialogView == null) {
            dialogView = createDialogViewInternal(instance);
        }
        return (T) dialogView;
    }

    private synchronized AbstractDialogView createDialogViewInternal(DialogInstance instance) {
        AbstractDialogView dialogView = loadDialogView(instance.getDialogConstant().getInternalName(), instance.getDialogInstanceNumber());
        dialogViews.put(instance, dialogView);
        dialogView.setApplicationView(this);
        if (getDialogController(instance) == null) {
            getApplicationController().createDialogInstance(instance.getDialogConstant(), instance.getDialogInstanceNumber());
        }
        JVineController.getInstance(getDialogController(instance)).registerView(dialogView);
        return dialogView;
    }

    private synchronized <T extends AbstractDialogView> T loadDialogView(String beanID, int instanceNumber) {
        beanID = beanID + AbstractApplicationView.VIEW;
        if (log.isDebugEnabled()) {
            log.debug("Creating new view using beanID " + beanID + " (instance " + instanceNumber + ")");
        }
        @SuppressWarnings("unchecked")
        T dialogView = (T) G9Spring.getBean(DialogView.class, beanID);
        dialogView.setDialogInstanceNumber(instanceNumber);
        return dialogView;
    }

    /**
     * Add a view to this application view.
     *
     * @param view the dialog view instance
     */
    synchronized void addDialogView(AbstractDialogView view){
    	dialogViews.put(view.getDialogInstance(), view);
    }


    /**
     * Get the dialog controller for the given dialog. Routes the request to the
     * application controller.
     *
     * @param dialogConstant the requested dialog.
     * @return the dialog controller via the application controller.
     */
    public DialogController getDialogController(DialogConstant dialogConstant) {
        return getApplicationController().getDialogController(dialogConstant);
    }

    /**
     * Get the dialog controller for the given dialog instance. Routes the
     * request to the application controller.
     *
     * @param dialogInstance the requested dialog instance.
     * @return the dialog controller via the application controller.
     */
    public DialogController getDialogController(DialogInstance dialogInstance) {
        return getApplicationController().getDialogController(dialogInstance);
    }

    /**
     * Get the name of the current dialog instance. If <code>doOpen</code>
     * is true, add the first dialog to the stack of open dialogs.
     *
     * @param doOpenFirst open the first dialog if no dialog is open
     *
     * @return the current dialog instance name
     */
    public String getCurrentDialogInstanceName(boolean doOpenFirst) {
        DialogInstance instance = getCurrentDialog(doOpenFirst);
        return getDialogInstanceName(instance);
    }

    /**
     * @param doOpenFirst open the first dialog if no dialog is open
     * @return the current (open) dialog. If no dialogs are opened, the
     * first dialog is opened (by adding it to the list of open dialogs).
     */
    public DialogInstance getCurrentDialog(boolean doOpenFirst) {
        if (openDialogs.isEmpty()) {
            DialogInstance firstDialog = getFirstDialogInstance();
            if (firstDialog != null && doOpenFirst) {
                openDialogs.add(firstDialog);
                triggerDialogEvent(firstDialog, WindowEvent.OPENED);
            } else {
                return firstDialog;
            }
        }
        return openDialogs.get(openDialogs.size() - 1);
    }

    /**
     * Get the first dialog in the application, ie. the start
     * dialog.
     *
     * @return the first dialog
     */
    protected abstract DialogConstant getFirstDialog();

    /**
     * Get the name of the empty dialog, i.e. the dialog which is "shown"
     * if there is no StartDialog parameter in the template parameters
     * when the application is generated.
     *
     * @return the name of the empty dialog
     */
    public abstract String getEmptyDialogName();

    /**
     * @return the current message.
     */
    public Message getMessage() {
        return message;
    }

    /**
     * Set the current message.
     *
     * @param message the new current message.
     */
    public void setMessage(Message message) {
        this.message = message;
    }

    /**
     * @return the current message callback.
     */
    public MessageCallback getMessageCallback() {
        return messageCallback;
    }

    /**
     * Set the current message callback.
     *
     * @param messageCallback the new current message callback.
     */
    public void setMessageCallback(MessageCallback messageCallback) {
        this.messageCallback = messageCallback;
    }

    /**
     * @return true if a message should be shown, false otherwise.
     */
    public boolean getShowMessage() {
        return message != null ? true : false;
    }

    /**
     * @param reply the message reply type.
     */
    public void handleMessageReply(MessageReplyType reply) {
        Message currentMsg = message;
        MessageCallback currentCb = messageCallback;
        message = null;
        messageCallback = null;
        currentMsg.setReply(reply);
        if (currentCb != null) {
            currentCb.reply(currentMsg);
        }
    }

    @Override
    public void performAction(final ActionType action,
            final DialogConstant target) {
        performAction(action, target, false);
    }

    @Override
    public void performAction(final ActionType action, final DialogConstant target, Boolean flag) {
        performAction(action, target, flag, false);
    }

    /**
     * Perform the specified action.
     *
     * @param action the action to perform
     * @param target the action's target
     * @param flag general purpose flag
     * @param fromExternal if true, the action source is external
     */
    public void performAction(final ActionType action, final DialogConstant target, Boolean flag, boolean fromExternal) {

        if (log.isTraceEnabled()) {
            log.trace("performAction: " + action + " " + target + ", flag: " + flag);
        }
        switch (action) {
            case OPEN:
                open(target, flag, fromExternal);
                break;
            case CLOSE:
                close(target);
                break;
            case SHOW:
                show(target);
                break;
            case HIDE:
                hide(target);
                break;
            case CLEAROBJECT:
                clearObject(target);
                break;
            default:
                if (log.isTraceEnabled()) {
                    log.trace("Application action not handled: " + action);
                }
                break;
        }
    }

    /**
     * Return the dialog name for the given dialog.
     * The name is used internally as a string id for the dialog.
     * NB! Not for public use, the actual name used may change!
     *
     * @param dialogConstant the dialog
     * @return the name of the dialog
     */
    public String getDialogName(DialogConstant dialogConstant) {
        return dialogConstant.getG9Name();
    }

    /**
     * Return the dialog instance name for the given dialog instance.
     * The name is used internally as a string id for the dialog instance.
     * NB! Not for public use, the actual name used may change!
     *
     * @param instance the dialog
     * @return the name of the dialog
     */
    public String getDialogInstanceName(DialogInstance instance) {
        if (instance == null) {
            return getEmptyDialogName();
        }
        String retVal = getDialogName(instance.getDialogConstant());
        if (instance.getDialogInstanceNumber() > 1) {
            retVal = retVal + instance.getDialogInstanceNumber();
        }
        return retVal;
    }

    private void triggerDialogEvent(DialogInstance dialogInstance, WindowEvent type) {
        AbstractDialogView view = getDialogView(dialogInstance);
        if (view != null) {
            view.triggerWindowEvent(type);
        }
    }

    /**
     * Trigger the correct dialog event as a result of the given action.
     * I an event method is modeled for the event, the event method will
     * be dispatched.
     * Note that this method should only be called from a worker thread,
     * and not the GUI thread.
     * Although this method is part of the public API, it is intended to
     * only be used from framework code.
     * 
     * @param target the target dialog
     * @param action the dialog action
     */
    public void triggerShownHiddenEvent(DialogConstant target, ActionType action) {
    	WindowEvent event = null;
        switch (action) {
        case SHOW:
        	event = WindowEvent.SHOWN;
            break;
        case HIDE:
        	event = WindowEvent.HIDDEN;
            break;
        default:
            break;
        }
        if (event != null) {
        	DialogInstance instance = getCurrentDialogInstance(target);
        	AbstractDialogView view = getDialogView(instance);
        	if (view != null) {
        		view.triggerWindowEvent(event);
        	}
        }
    }
    
    @Override
    public boolean isOpen(DialogConstant target) {
        DialogInstance instance = getCurrentDialogInstance(target);
        if (instance != null) {
            if (openDialogs.contains(instance) || openDialogBoxes.contains(instance)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean isShown(DialogConstant target) {
        DialogInstance instance = getCurrentDialogInstance(target);
        if (instance != null) {
            return isShown(instance);
        }
        return false;
    }

    private boolean isShown(DialogInstance instance) {
        if (openDialogs.contains(instance)) {
            AbstractDialogView view = getDialogView(instance);
            return view.isShown();
        }
        if (openDialogBoxes.contains(instance)) {
            Boolean dialogBoxShown = shown.get(instance);
            if (dialogBoxShown != null && dialogBoxShown.booleanValue()) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void open(final DialogConstant dialogConstant) {
        open(dialogConstant, false, false);
    }

    @Override
    public void open(final DialogConstant dialogConstant, final boolean newDialog) {
        open(dialogConstant, newDialog, false);
    }

    /**
     * Open the given dialog.
     *
     * @param dialogConstant the dialog to open
     * @param newDialog if <code>true</code> a <em>new</em> instance of the dialog will be created.
     * @param fromExternal if true, the action source was external
     */
    private void open(final DialogConstant dialogConstant, final boolean newDialog, final boolean fromExternal) {
        guiInvoke(new GuiTask("open") {
            @Override
            public void run() {
                DialogInstance instance = getDialogInstance(dialogConstant, newDialog, fromExternal);
                if (dialogConstant.getWindowType() == WindowType.DIALOG_BOX) {
                    if (!openDialogBoxes.contains(instance)) {
                        openDialogBoxes.add(instance);
                        if (instance instanceof DialogInstanceKeyFromExternal) {
                            sendPushMessage();
                        }
                    }
                    show(instance); // Opening a dialog box implies "show"
                } else {
                    if (openDialogs.contains(instance)) {
                        sendElevateMessage();
                    } else {
                        sendPushMessage();
                    }
                    openDialogs.remove(instance);
                    openDialogs.add(instance);
                }
                triggerDialogEvent(instance, WindowEvent.OPENED);
            }
        });

    }

    @Override
    public void close(final DialogConstant dialogConstant) {
        guiInvoke(new GuiTask("close") {
            @Override
            public void run() {
                DialogInstance instance = getCurrentDialogInstance(dialogConstant);
                if (instance != null) {
                    if (openDialogs.contains(instance)) {
                        instance = openDialogs.get(openDialogs.indexOf(instance));
                    }
                    else if (openDialogBoxes.contains(instance)) {
                        instance = openDialogBoxes.get(openDialogBoxes.indexOf(instance));
                    }
                }
                if (instance != null) {
                    if(log.isTraceEnabled()) {
                        log.trace("closing " + dialogConstant);
                    }
                    if (openDialogs.remove(instance) || openDialogBoxes.remove(instance)) {
                        sendElevateMessage();
                        sendPopMessage();
                    }
                    triggerDialogEvent(instance, WindowEvent.CLOSED);
                    clearObject(instance);
                    removeInstance(instance);
                }
            }
        });

    }

    /**
     * Removes the instance from the view maps and the instance Stack.
     * Also signal the system if the dialog instance was opened from an external source.
     *
     * @param instance the instance to remove
     */
    public void removeInstance(DialogInstance instance) {
        dialogViews.remove(instance);
        viewModels.remove(instance);
        getApplicationController().removeDialogInstance(instance);
    }

    @Override
    public void show(final DialogConstant dialogConstant) {
        guiInvoke(new GuiTask("show") {
            @Override
            public void run() {
                DialogInstance instance = getCurrentDialogInstance(dialogConstant);
                if (instance != null) {
                    show(instance);
                }
            }
        });
    }

    /**
     * Show the given dialog instance.
     *
     * @param instance the dialog to show
     */
    protected void show(DialogInstance instance) {
        DialogInstance stackInstance = null;
        if (openDialogs.contains(instance)) {
            stackInstance = openDialogs.get(openDialogs.indexOf(instance));
            openDialogs.remove(stackInstance);
            openDialogs.add(stackInstance);
            //triggerDialogEvent(instance, WindowEvent.SHOWN); NB! This is now triggered from the action task
        } else if (openDialogBoxes.contains(instance)) {
            shown.put(instance, Boolean.TRUE);
            //triggerDialogEvent(instance, WindowEvent.SHOWN); NB! This is now triggered from the action task
        }
        sendElevateMessage();
        getApplicationController().setActiveDialogInstance(instance);
    }

    @Override
    public void hide(final DialogConstant dialogConstant) {
        guiInvoke(new GuiTask("hide") {
            @Override
            public void run() {
                DialogInstance instance = getCurrentDialogInstance(dialogConstant);
                if (instance != null) {
                    if (openDialogs.contains(instance)) {
                        DialogInstance stackInstance = openDialogs.get(openDialogs.indexOf(instance));
                        openDialogs.remove(stackInstance);
                        openDialogs.add(0, stackInstance);
                        //triggerDialogEvent(instance, WindowEvent.HIDDEN); NB! This is now triggered from the action task
                    } else if (openDialogBoxes.contains(instance)) {
                        shown.put(instance, Boolean.FALSE);
                        //triggerDialogEvent(instance, WindowEvent.HIDDEN); NB! This is now triggered from the action task
                    }
                }
            }
        });
    }

    @Override
    public void clearObject(final DialogConstant dialogConstant) {
        guiInvoke(new GuiTask("clearObject") {
            @Override
            public void run() {
                DialogInstance instance = getCurrentDialogInstance(dialogConstant);
                if (instance != null) {
                    clearObject(instance);
                }
            }
        });
    }

    private void clearObject(DialogInstance instance) {
        AbstractDialogView view = getDialogView(instance);
        if (view != null) {
            view.clearDialog();
        }
    }

    /**
     * @return the list of currently open dialogs.
     */
    protected List<DialogInstance> getOpenDialogs() {
        return openDialogs;
    }

    /**
     * @return the list of currently open (and shown) dialog boxes.
     */
    public List<DialogInstance> getOpenDialogBoxes() {
        List<DialogInstance> boxes = new ArrayList<DialogInstance>();
        for (DialogInstance dialog : openDialogBoxes) {
            if (isShown(dialog)) {
                boxes.add(dialog);
            }
        }
        return boxes;
    }

    /**
     * Returns a dispatcher context that is set up with "this" application view.
     *
     * @return a dispatcher context for this application view.
     */
    public JVineDispatcherContext getDispatcherContext() {

        BlockingMessageCallback callBack = getBlockinMessageCallback();
        return getDispatcherContext(callBack);
    }

    private BlockingMessageCallback getBlockinMessageCallback() {
        BlockingMessageCallback callBack = ServiceLoader.getService(BlockingMessageCallback.class);
        if (callBack instanceof UAMessageCallback) {
            UAMessageCallback uaMsgCb = (UAMessageCallback) callBack;
            JVineAppController jAppCtrl = JVineAppController.getInstance(applicationController);
            uaMsgCb.setGuiActionQueue(jAppCtrl.getActionQueue());
        }
        return callBack;
    }

    /**
     * Returns a <code>DispatcherContext</code> to be used for dispatching messages to this application.
     * @param messageCallback The callback used in this context.
     * @return The context.
     */
    public JVineDispatcherContext getDispatcherContext(final MessageCallback messageCallback) {
        return new JVineDispatcherContext() {

            @Override
            public AbstractApplicationView getApplicationView() {
                return AbstractApplicationView.this;
            }

            @Override
            public MessageCallback getMessageCallback() {
               return messageCallback;
            }

            @Override
            public String toString() {
                return "JVineDispatcherContext ["
                        + getApplicationController().getApplicationName() + "]";
            }

            @Override
            public InteractionThreadPolicy getInteractionThreadPolicy() {
                return messageCallback.getInteractionThreadPolicy();
            }



        };
    }

    /**
     * @return a message dispatcher with a default dispatcher context.
     */
    public MessageDispatcher getMessageDispatcher() {
        return getMessageDispatcher(getDispatcherContext());
    }

    /**
     * @param dispatcherContext the dispatcher context
     * @return a message dispatcher with the specified dispatcher context
     */
     public MessageDispatcher getMessageDispatcher(
            DispatcherContext dispatcherContext) {
        return MessageSystem.getMessageDispatcher(dispatcherContext);
    }

    /**
     * Ensure that the given method is run on the GUI thread.
     *
     * @param method the method to run
     */
    protected void guiInvoke(GuiTask method) {
        if (ThreadManager.isWorkerThread()) {
            try {
                getApplicationController().invokeOnGui(method);
            } catch (InvocationTargetException e) {

                MessageDispatcher messageDispatcher = getMessageDispatcher();
                Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_CAUGHT, method.toString());
                msg.setReplString("OK");
                msg.setException(e);

                messageDispatcher.dispatch(CRuntimeMsg.CF_CAUGHT, msg);
            }
        } else {
            method.run();
        }
    }

    private DialogInstance getFirstDialogInstance() {
        DialogConstant dialog = getFirstDialog();
        if (getDialogName(dialog).equals(getEmptyDialogName())) {
            return null;
        }
        DialogInstance instance = getApplicationController().getCurrentDialogInstance(dialog);
        if (instance == null) {
            instance = getApplicationController().createDialogInstance(getFirstDialog(), -1);
        }
        return instance;
    }

    private DialogInstance getCurrentDialogInstance(DialogConstant dialog) {
        return getApplicationController().getCurrentDialogInstance(dialog);
    }

    private DialogInstance getDialogInstance(DialogConstant dialog, boolean newInstance, boolean fromExternal) {
        DialogInstance instance = null;
        if (!newInstance) {
            instance = getCurrentDialogInstance(dialog);
        }
        if (instance == null) {
            instance = getApplicationController().createDialogInstance(dialog, -1);
        }
        if (fromExternal) {
            instance = new DialogInstanceKeyFromExternal(instance);
        }
        return instance;
    }

    /**
     * Get the title for the given dialog instance.
     * If the dialogTitle map contains a title for the
     * given instance, this title will be used. Otherwise
     * the default dialog title will be used.
     *
     * @param instance the dialog instance in question
     * @return the title for the instance
     */
    @Override
    public String getDialogTitle(DialogInstance instance) {
        String title = dialogTitle.get(instance);
        if (title == null) {
            title = instance.getDialogConstant().getTitle();
            if (instance.getDialogInstanceNumber() > 1) {
                title = title + " #" + instance.getDialogInstanceNumber();
            }
        }
        return title;
    }

    /**
     * Set the title for the given dialog instance.
     * If the title parameter is null, the instance is removed
     * from the dialogTitle map, and the default dialog title
     * will be used instead.
     */
    @Override
    public void setDialogTitle(DialogInstance instance, String title) {
        if (title == null) {
            dialogTitle.remove(instance);
        }
        else {
            dialogTitle.put(instance, title);
        }
    }

    /**
     * Get the current system message queue.
     * Note: This method is *not* part of the public API!
     *
     * @return the systemMessages
     */
    public List<SystemMessage> getOutputMessages() {
        return outputMessages;
    }

    /**
     * Notify the system that we want to be on top of the application stack
     */
    public void sendPushMessage() {
        String applicationName = getApplicationController().getApplicationName();
        SystemMessage pushMessage = SystemMessage.STACK_PUSH_TEMPLATE.payload(new SystemMessage(applicationName, "", ""));
        getOutputMessages().add(pushMessage);
    }

    /**
     * Notify the system that we want to be on top of the application stack,
     * by moving an exiting entry to the stack top.
     */
    public void sendElevateMessage() {
        String applicationName = getApplicationController().getApplicationName();
        SystemMessage elevateMessage = SystemMessage.STACK_ELEVATE_TEMPLATE.payload(new SystemMessage(applicationName, "", ""));
        getOutputMessages().add(elevateMessage);
    }

    /**
     * Notify the system that we want to be popped from the application stack
     */
    public void sendPopMessage() {
        getOutputMessages().add(SystemMessage.STACK_POP_TEMPLATE);
    }

    /**
     * Check for any dirty views.
     * 
     * @return true if a view is dirty, else false.
     */
    public boolean hasDirtyDialogs() {
        for (ViewModel viewModel : viewModels.values()) {
            if (viewModel.getChangedFields().size() > 0) {
                DialogController ctrl= viewModel.getDialogController();
                ArrayList<DialogObjectConstant> changedFields= Lists.newArrayList();
                for (DialogObjectConstant field : viewModel.getChangedFields()) {
                    if (field.getAttribute() != null && field.getAttribute().getAttributeRole() != null) {
                        RoleConstant ar= field.getAttribute().getAttributeRole();
                        // Call checkChange in case of interceptors
                        if (!(ar instanceof UsertypeRoleConstant) && !ctrl.checkChange(ar, CheckType.CLOSE)) {
                            changedFields.add(field);
                        }
                    }
                }
                
                if (changedFields.size() > 0) {
                    if (log.isDebugEnabled()) {
                        log.debug("Dirty fields for dialog "+ctrl.getDialogConstant().getG9Name()+":");
                        for (DialogObjectConstant field : changedFields) {
                            log.debug("Dirty field: "+field.getInternalName()+" (G9Name= "+field.getG9Name()+")");
                        }
                    }
                    return true;
                }
            }
        }
        return false;
    }
}
