/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.view;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;

import no.esito.jvine.action.GuiTask;
import no.esito.jvine.action.ThreadManager;
import no.esito.jvine.controller.DialogInstanceKey;
import no.esito.jvine.controller.JVineApplicationController;
import no.esito.jvine.controller.JVineController;
import no.esito.jvine.controller.OSNode;
import no.esito.jvine.util.ViewConstantHelper;
import no.esito.log.Logger;
import no.g9.client.core.action.CheckType;
import no.g9.client.core.action.EventContext;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.ListSelectionCallback;
import no.g9.client.core.view.ViewModel;
import no.g9.client.core.view.WindowEvent;
import no.g9.client.core.view.table.TableModel;
import no.g9.client.core.view.table.TableModel.SelectionModel;
import no.g9.client.core.view.tree.TreeModel;
import no.g9.client.core.view.tree.TreeNode;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageDispatcher;
import no.g9.message.MessageSystem;
import no.g9.os.AttributeConstant;
import no.g9.os.RoleConstant;
import no.g9.service.G9Spring;
import no.g9.support.ActionType;
import no.g9.support.convert.ConvertException;

/**
 * Common superclass for all generated views for the dialogs in the application.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 *
 */
public abstract class AbstractDialogView implements DialogView {

    private DialogInstance instance;

    /** Set from the constructor. The <code>instance</code> is set during initialization */
    private DialogConstant tmpDialog;

    private int currentRowNum;

    private boolean isOpen;

    private boolean isShown;

    private boolean modal;

    private AbstractApplicationView applicationView;

    private Map<DialogObjectConstant, DialogObjectConstant> currentNotebookPage = new HashMap<DialogObjectConstant, DialogObjectConstant>();

    private Map<DialogObjectConstant, Map<ActionType, String>> effects = new HashMap<DialogObjectConstant, Map<ActionType, String>>();

    /** The default button as defined in the dialog model, or as set from the program */
    private DialogObjectConstant defaultButton;

    /**
     * Common logger for this and its subclasses.
     */
    protected static final Logger log = Logger.getLogger(DialogView.class);

    /**
     * Create a new view for the given dialog.
     * NB! The dialog view needs an instance number before it can be used by the application!
     *
     * @param dialog the dialog which owns this view.
     */
    public AbstractDialogView(DialogConstant dialog) {
        this.tmpDialog = dialog;
        this.modal = false;
    }

    @Override
    public void update(Collection<OSNode<?>> nodes) {
        if (log.isTraceEnabled()) {
            log.trace("Updating  " + nodes);
        }
        for (OSNode<?> node : nodes) {
            Collection<?> instances = JVineController.getInstance(
                    getDialogController()).getAllInstances(node);
            setAllFieldData(node, instances);
        }
    }

    @Override
    public void performAction(ActionType action, DialogObjectConstant target) {
        if (log.isTraceEnabled()) {
            log.trace("performAction: " + action + " " + target);
        }
        Object effectBean = getEffectBean(target, action);
        if (effectBean != null) {
            if (log.isTraceEnabled()) {
                log.trace("performAction, setting effect: "
                        + getEffectBeanId(target, action));
            }
            ViewModelImpl model = (ViewModelImpl) getViewModel();
            model.getPropertyManager().setEffect(target, effectBean);
        }
        switch (action) {
        case HIDE:
            // We assume the effect will set the "shown" property
            if (effectBean == null) {
                hide(target);
            }
            break;
        case SHOW:
            // We assume the effect will set the "shown" property
            if (effectBean == null) {
                show(target);
            }
            break;
        case ENABLE:
            enable(target);
            break;
        case DISABLE:
            disable(target);
            break;
        case CLEAROBJECT:
            clearObject(target);
            break;
        case FOCUS:
            focus(target);
            break;
        case UNSELECT:
            unselect(target);
            break;
        case SELECT:
            select(target);
            break;
        default:
            if (log.isTraceEnabled()) {
                log.trace("Dialog action not handled: " + action);
            }
            break;
        }
    }

    private void select(DialogObjectConstant target) {
        ViewModelImpl view = (ViewModelImpl) getViewModel();
        RoleConstant role= target.getRole();
        view.setListRowSelection(true, role, this.currentRowNum, null);
    }

    private void unselect(DialogObjectConstant target) {
        ViewModelImpl view = (ViewModelImpl) getViewModel();
        RoleConstant role= target.getRole();
        view.setListRowSelection(false, role, this.currentRowNum, null);
    }

    @Override
    public boolean isEditable(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        return model.getPropertyManager().isEditable(target);
    }

    @Override
    public boolean isEnabled(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        return model.getPropertyManager().isEnabled(target);
    }

    @Override
    public String getTitle(DialogObjectConstant dialogObject) {
        String messageID = dialogObject.getMessageID();

        if (messageID == null) {
            return null;
        }

        ResourceBundle bundle = ResourceBundle.getBundle(getMessageBundleName());
        return bundle.getString(messageID);
    }

    @Override
    public void setDialogTitle(String title) {
        getApplicationView().setDialogTitle(getDialogInstance(), title);
    }

    /**
     * Get the message bundle name
     * @return the message bundle name
     *
     */
    public abstract String getMessageBundleName();

    @Override
    public boolean isMandatory(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        return model.getPropertyManager().isMandatory(target);
    }

    @Override
    public boolean isShown(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        return model.getPropertyManager().isShown(target);
    }

    @Override
    public void setModal(boolean modal) {
        this.modal = modal;
    }

    @Override
    public boolean isModal() {
        return modal;
    }

    @Override
    public Object getEffect(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        return model.getPropertyManager().getEffect(target);
    }

    @Override
    public void setEditable(DialogObjectConstant target, boolean value) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.getPropertyManager().setEditable(target, value);
    }

    @Override
    public void setMandatory(DialogObjectConstant target, boolean value) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.getPropertyManager().setMandatory(target, value);
    }

    @Override
    public boolean getMandatory(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        return model.getPropertyManager().getMandatory(target);
    }

    @Override
    public void setEffect(DialogObjectConstant target, Object effect) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.getPropertyManager().setEffect(target, effect);
    }

    @Override
    public void enable(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.getPropertyManager().setEnabled(target, true);
    }

    @Override
    public void disable(final DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.getPropertyManager().setEnabled(target, false);
    }

    @Override
    public void show(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.getPropertyManager().setShown(target, true);
    }

    @Override
    public void hide(DialogObjectConstant target) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.getPropertyManager().setShown(target, false);
    }

    @Override
    public void setWidgetProperty(DialogObjectConstant target, String key, Object value) {
        if (target != null) {
            ViewModelImpl model = (ViewModelImpl) getViewModel();
            model.getPropertyManager().setProperty(target, key, value);
        }
    }

    @Override
    public Object getWidgetProperty(DialogObjectConstant target, String key) {
        if (target != null) {
            ViewModelImpl model = (ViewModelImpl) getViewModel();
            return model.getPropertyManager().getProperty(target, key);
        }
        return null;
    }

    @Override
    public void setCurrentNotebookPage(DialogObjectConstant notebook,
            DialogObjectConstant page) {
        currentNotebookPage.put(notebook, page);
    }

    @Override
    public DialogObjectConstant getCurrentNotebookPage(
            DialogObjectConstant notebook) {
        return currentNotebookPage.get(notebook);
    }

    @Override
    public ViewModel getViewModel() {
        return getApplicationView().getViewModel(instance);
    }

    /**
     * Get the dialog constant (enum) for this view.
     *
     * @return the dialog constant.
     */
    public DialogConstant getDialog() {
        return instance.getDialogConstant();
    }

    /**
     * @return the dialog instance for this view
     */
    public DialogInstance getDialogInstance() {
        return instance;
    }

    /**
     * Get the dialog controller for this view.
     *
     * @return the dialog controller.
     */
    protected DialogController getDialogController() {
        return applicationView.getDialogController(instance);
    }

    @Override
    public int getDialogInstanceNumber() {
        return instance.getDialogInstanceNumber();
    }

    /**
     * Set the dialog instance number for this view. This creates a dialog instance identifier
     * for this dialog, which is essential before using the dialog!
     *
     * @param dialogInstanceNumber the unique instance number for this dialog view
     */
    protected void setDialogInstanceNumber(int dialogInstanceNumber) {
        this.instance = new DialogInstanceKey(tmpDialog, dialogInstanceNumber);
    }

    /**
     * Get the application view, which references the application controller.
     *
     * @return the application view.
     */
    public AbstractApplicationView getApplicationView() {
        return applicationView;
    }

    /**
     * Set the application view.
     *
     * @param applicationView
     *            the new application view.
     */
    public void setApplicationView(AbstractApplicationView applicationView) {
        this.applicationView = applicationView;
        initView();
    }

    /**
     * Initialize the view state. This method is called after the application
     * bean has been set.
     */
    protected void initView() {
        init(); // The initialization method to override in generated subclasses
    }

    /**
     * Set field values for all the given instances. Invariant: only
     * many-related nodes (including the root node) in the object selection may
     * be given for the node parameter.
     *
     * @param node
     *            the many-related node for the given instances.
     * @param instances
     *            the collection of instances with the field values.
     */
    public abstract void setAllFieldData(OSNode<?> node, Collection<?> instances);

    /**
     * Set the dialog object to receive focus.
     *
     * @param target
     *            the dialog object to focus.
     */
    protected abstract void focus(final DialogObjectConstant target);

    @Override
    public void dispatchEvent(ListRow listRow, DialogObjectConstant trigger, Object event, String method) {
        if (canDispatchEvents()) {
        	setThreadApplicationController();
            EventContext eventMethod = new EventContext(method, trigger, event, listRow);
            JVineController.getInstance(getDialogController()).dispatch(eventMethod);
        }
    }

    /**
     * Check if the dialog can dispatch events.
     * If the application is waiting for user response in the message box, the event can not be dispatched.
     * Also, the dialog must be open and shown to process events.
     *
     * @return true if the dialog can dispatch events, false otherwise
     */
    public boolean canDispatchEvents() {
        return !applicationView.getShowMessage() && isOpen() && isShown() && (getDialogController() != null);
    }

    /**
     * Dispatch the event without a listRow.
     *
     * @param trigger the cause of the event.
     * @param type the type of event.
     * @param method the event method name.
     */
    protected void dispatchEvent(DialogObjectConstant trigger, String type, String method) {
        dispatchEvent(null, trigger, type, method);
    }

    /**
     * Trigger events for the window block. The dialog views must override this
     * method with the actual event method to be used.
     *
     * @param type
     *            the event type.
     */
    protected abstract void triggerWindowEvent(WindowEvent type);

    /**
     * Clear all fields of the dialog.
     */
    public void clearDialog() {
        getViewModel().clearViewModel();
    }

    @Override
    public void clearObject(DialogObjectConstant target) {
        if (isListComponent(target)) {
            RoleConstant listRole = getListRole(target);
            getViewModel().getTableModel(listRole).clear();
        } else {

            AttributeConstant attribute = target.getAttribute();
            if (attribute != null) {
                getViewModel().setFieldValue(target, null);
            }
        }

        Collection<DialogObjectConstant> children = ViewConstantHelper
                .getChildren(target);
        for (DialogObjectConstant childTarget : children) {
            clearObject(childTarget);
        }
    }

    /**
     * Get the bean ID for the effect for the given dialog object and action
     * type.
     *
     * @param target
     *            the dialog object which has the effect.
     * @param action
     *            the action type for the effect.
     * @return the bean ID if found, or null.
     */
    public String getEffectBeanId(DialogObjectConstant target, ActionType action) {
        String effectBeanId = null;
        Map<ActionType, String> targetEffects = effects.get(target);
        if (targetEffects != null) {
            effectBeanId = targetEffects.get(action);
        }
        return effectBeanId;
    }

    /**
     * Set the bean ID for the effect for the given dialog object and action
     * type.
     *
     * @param target
     *            the dialog object which has the effect.
     * @param action
     *            the action type for the effect.
     * @param effectBeanId
     *            the bean ID for the effect.
     */
    public void setEffectBeanId(DialogObjectConstant target, ActionType action,
            String effectBeanId) {
        Map<ActionType, String> targetEffects = effects.get(target);
        if (targetEffects == null) {
            targetEffects = new HashMap<ActionType, String>();
            effects.put(target, targetEffects);
        }
        targetEffects.put(action, effectBeanId);
    }

    /**
     * Get the bean for the effect for the given dialog object and action type.
     *
     * @param target
     *            the dialog object which has the effect.
     * @param action
     *            the action type for the effect.
     * @return the bean if found, or null.
     */
    public Object getEffectBean(DialogObjectConstant target, ActionType action) {
        String effectBeanId = getEffectBeanId(target, action);
        if (effectBeanId != null) {
            return G9Spring.getBean(effectBeanId);
        }
        return null;
    }

    @Override
    public DialogObjectConstant getDefaultButton() {
        return defaultButton;
    }

    @Override
    public void setDefaultButton(DialogObjectConstant defaultButton) {
        this.defaultButton = defaultButton;
    }

    /**
     * Test if the specified dialog object represents a list component (e.g.
     * list block). If the dialog object represents a list component, the role
     * of the list component is returned.
     *
     * @param target
     *            the dialog object constant.
     * @return the role of the list component or <code>null</code>.
     */
    private boolean isListComponent(DialogObjectConstant target) {
        Map<DialogObjectConstant, RoleConstant> listRoles = getListRoles();
        if (listRoles != null) {
            return listRoles.keySet().contains(target);
        }
        return false;
    }

    /**
     * Get the role of the specified list component.
     *
     * @param listComponent
     *            the list component
     * @return the role of the list component or <code>null</code>.
     */
    private RoleConstant getListRole(DialogObjectConstant listComponent) {
        Map<DialogObjectConstant, RoleConstant> listRoles = getListRoles();
        if (listRoles != null) {
            return listRoles.get(listComponent);
        }

        return null;
    }

    /**
     * Get the map from list components to roles.
     *
     * @return the map from list components to roles.
     */
    private Map<DialogObjectConstant, RoleConstant> getListRoles() {
        ViewModel viewModel = getViewModel();
        if (viewModel instanceof ViewModelImpl) {
            ViewModelImpl viewModelImpl = (ViewModelImpl) viewModel;
            return viewModelImpl.getListRoles();
        }
        return null;
    }

    /**
     * @return the open status of the dialog.
     */
    public boolean isOpen() {
        return isOpen;
    }

    /**
     * @param isOpen the open status to set.
     */
    protected void setOpen(boolean isOpen) {
        this.isOpen = isOpen;
    }

    /**
     * @return the shown status of the dialog.
     */
    protected boolean isShown() {
        return isShown;
    }

    /**
     * @param isShown the shown status to set.
     */
    protected void setShown(boolean isShown) {
        this.isShown = isShown;
    }

    /**
     * Delegate method for the view model.
     *
     * @param field The (generated) enum denoting the role attribute
     * @return the specified field value (possibly converted from view to model value)
     */
    protected Object getFieldValue(DialogObjectConstant field) {
        return getViewModel().getFieldValue(field);
    }

    /**
     * Delegate method for the view model.
     *
     * @param field The (generated) enum denoting the role attribute
     * @param fieldValue The field value to display (possibly converted from model to view value)
     */
    protected void setFieldValue(DialogObjectConstant field, Object fieldValue) {
        getViewModel().setFieldValue(field, fieldValue);
    }

    /**
     * Delegate method for the view model.
     *
     * @param role
     *            the role which owns the field collection.
     * @return the field collection.
     */
    protected Collection<DialogObjectConstant> getRoleFields(RoleConstant role) {
        return getViewModel().getRoleFields(role);
    }

    /**
     * Delegate method for the view model.
     *
     * @param role
     *            the role which owns the attribute.
     * @param field
     *            the field to add.
     */
    protected void addRoleField(RoleConstant role, DialogObjectConstant field) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.addRoleField(role, field);
    }

    /**
     * Delegate method for the view model.
     *
     * @param role
     *            the role which owns the table model
     * @param listConst
     *            the list constant for the table model
     * @param tableModel
     *            the list to add
     */
    protected void addRoleList(RoleConstant role,
    		DialogObjectConstant listConst,
    		TableModel<? extends ListRow> tableModel) {
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.addRoleTableModel(role, listConst, tableModel);
    }

    /**
     * Delegate method for the view model.
     *
     * @param treeConst
     *            the role for the treeview
     * @param treeModel
     *            the model of the treeview
     */
    protected void addTreeModel(DialogObjectConstant treeConst, TreeModel<? extends TreeNode, ? extends ListRow> treeModel) {
    	ViewModelImpl model = (ViewModelImpl) getViewModel();
        model.setTreeModel(treeConst, treeModel);
    }

    /**
     * Handle a select/unselect for a list row. If the edit fields for the list
     * has changed, a message box is shown asking the user to proceed or not.
     *
     * @param listRole
     *            - the object selection role for the list
     * @param selected
     *            - true if the row is to be selected
     * @param rowNo
     *            - the list row in question
     * @param emptyRow
     *            - an empty list row containing a collection of all the fields
     *            in a list row
     * @return true if the selection in the list was changed, false if the
     *         operation was canceled by the user
     */
    protected boolean handleListSelection(RoleConstant listRole,
            boolean selected, int rowNo, ListRow emptyRow) {

        boolean listSelectionChanged = true;
        log.debug("Handle list selection");
        // Show the message only if the edit fields have changed
        boolean unchanged = getDialogController().checkChange(listRole,
                CheckType.ROW_SELECT);
        ViewModelImpl model = (ViewModelImpl) getViewModel();
        SelectionModel selectionModel = model.getTableModel(listRole).getSelectionModel();
        if (unchanged || selectionModel == SelectionModel.NO_SELECT) {
            log.debug("List row is unchanged.");
            listSelectionChanged = model.setListRowSelection(selected,
                    listRole, rowNo, emptyRow);
            log.trace("Done setting list selection to model. "
                    + "List selection is "
                    + (listSelectionChanged ? " changed" : "uncanged"));
        } else {
            log.debug("List row is changed.");
            ListSelectionCallback cb = new ListSelectionCallback(model,
                    selected, listRole, rowNo, emptyRow);
            String msgNo = selected ? CRuntimeMsg.CC_CHECK_SELECTION_MSG
                    : CRuntimeMsg.CC_CHECK_UNSELECTION_MSG;
            MessageDispatcher messageDispatcher = applicationView
                    .getMessageDispatcher(applicationView.getDispatcherContext(cb));
            messageDispatcher.dispatch(msgNo, (Object[]) null);

            listSelectionChanged = false;

            log.trace("Asuming no list selection since message "
                    + "interaction with end user is asynchronous.");
        }
        return listSelectionChanged;
    }


    /**
     * Setter for the temporary list selection. Used
     * later by action Select.
     *
     * @param rowNum row number
     */
    protected void setListSelectionRowNum(int rowNum) {
        this.currentRowNum= rowNum;
    }

    /**
     * Invoke a task on the gui thread.
     *
     * @param method
     *            the GuiTask to invoke.
     */
    protected void guiInvoke(GuiTask method) {
        if (ThreadManager.isWorkerThread()) {
            try {
                getDialogController().invokeOnGui(method);
            } catch (InvocationTargetException e) {
                MessageDispatcher messageDispatcher = applicationView.getMessageDispatcher();
                Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_CAUGHT, method.toString());
                msg.setReplString("OK");
                msg.setException(e);
                messageDispatcher.dispatch(CRuntimeMsg.CF_CAUGHT, msg);
            }
        } else {
            method.run();
        }
    }

    /**
     * Convert the specified model value to a view value.
     * @param field generated enum denoting the field which displays the value
     * @param modelValue the value to be converted
     * @return the converted view value
     */
    protected Object convertToView(DialogObjectConstant field, Object modelValue) {
        ViewModelImpl viewModel = (ViewModelImpl) getViewModel();
        Object viewValue = modelValue;
        if (ViewModelImpl.hasConverter(field)) {
            try {
                viewValue = viewModel.convertToViewInternal(field, modelValue);
            } catch (ConvertException e) {
                JVineController.getInstance(getDialogController())
                .addConverterException(e);            }
        }
        return viewValue;
    }

    private void setThreadApplicationController() {
    	JVineApplicationController.setCurrentApplicationController(getApplicationView().getApplicationController());
    }

}
