/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;

import no.esito.jvine.controller.FieldData;
import no.esito.jvine.controller.JVineController;
import no.esito.jvine.validation.ValidationManager;
import no.esito.jvine.validation.ValidationManagerFactory;
import no.esito.log.Logger;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.converter.FieldConvertContext;
import no.g9.client.core.converter.JodaDateTimeConverter;
import no.g9.client.core.converter.LocalDateConverter;
import no.g9.client.core.converter.LocalDateTimeConverter;
import no.g9.client.core.converter.LocalTimeConverter;
import no.g9.client.core.converter.NumberConverter;
import no.g9.client.core.util.DialogObjectConstantHelper;
import no.g9.client.core.validator.FieldValidator;
import no.g9.client.core.validator.ValidateContext;
import no.g9.client.core.validator.ValidationPolicy.Policy;
import no.g9.client.core.validator.ValidationResult;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.ViewModel;
import no.g9.client.core.view.table.TableModel;
import no.g9.client.core.view.tree.TreeModel;
import no.g9.client.core.view.tree.TreeNode;
import no.g9.os.AttributeConstant;
import no.g9.os.RoleConstant;
import no.g9.service.G9Spring;
import no.g9.support.TypeTool;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertException;

/**
 * The view model implementation.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public class ViewModelImpl implements ViewModel {

    private static final Logger log = Logger.getLogger(ViewModel.class);

    private DialogInstance instance;

    private AbstractApplicationView applicationView;

    private Map<DialogObjectConstant, Object> fields =
            new HashMap<DialogObjectConstant, Object>();

    private Map<RoleConstant, Collection<DialogObjectConstant>> fieldsForRole =
            new HashMap<RoleConstant, Collection<DialogObjectConstant>>();

    private Set<DialogObjectConstant> changedFields =
            new HashSet<DialogObjectConstant>();

    private Map<RoleConstant, TableModel<? extends ListRow>> tableModels =
            new HashMap<RoleConstant, TableModel<? extends ListRow>>();

    private Map<DialogObjectConstant, TreeModel<? extends TreeNode, ? extends ListRow>> treeModels = new HashMap<>();

    private Map<DialogObjectConstant, RoleConstant> listRoles =
            new HashMap<DialogObjectConstant, RoleConstant>();

    private Map<RoleConstant, Integer> listSelections =
            new HashMap<RoleConstant, Integer>();

    private Map<DialogObjectConstant, FieldConvertContext> convertContexts =
            new HashMap<DialogObjectConstant, FieldConvertContext>();

    private Map<DialogObjectConstant, ValidateContext> validateContexts =
            new HashMap<DialogObjectConstant, ValidateContext>();

    private DefaultPropertyManager propertyManager =
            new DefaultPropertyManager();

    private ValidationManager validationManager;

    /**
     * Create a new ViewModelImpl for the given dialog. The new object also has
     * a reference to the application view to be able to get access to the
     * dialog controller.
     *
     * @param dialog
     *            the dialog which owns this model.
     * @param applicationView
     *            the view for the application.
     */
    public ViewModelImpl(DialogInstance dialog, AbstractApplicationView applicationView) {
        this.instance = dialog;
        this.applicationView = applicationView;
    }

    /**
     * @return the map of all fields in the model.
     */
    public Map<DialogObjectConstant, Object> getFields() {
        return fields;
    }

    /**
     * @return the map of all list roles in the model.
     */
    public Map<DialogObjectConstant, RoleConstant> getListRoles() {
        return listRoles;
    }

    /**
     * Get the field (view model value) for the given field.
     *
     * @param field
     *            the field holds the value for this field.
     * @return the field.
     */
    public Object getField(DialogObjectConstant field) {
        return getFields().get(field);
    }

    /**
     * Set the field (view model value) for the given field. If the new value
     * differs from the old value, the field is marked as changed.
     *
     * @param field
     *            the field holds the value for this field.
     * @param value
     *            the new field value.
     */
    public void setField(DialogObjectConstant field, Object value) {
        Object oldValue = getField(field);
        if (TypeTool.viewFieldDiffer(value, oldValue)) {
            setChanged(field, true);
        }
        getFields().put(field, value);
    }

    /**
     * Add the given field to the collection of fields for the given role.
     *
     * @param role
     *            the role which owns the attribute.
     * @param field
     *            the field to add.
     */
    public void addRoleField(RoleConstant role, DialogObjectConstant field) {
        Collection<DialogObjectConstant> roleFields = fieldsForRole.get(role);
        if (roleFields == null) {
            roleFields = new HashSet<DialogObjectConstant>();
            fieldsForRole.put(role, roleFields);
        }
        roleFields.add(field);
    }

    /**
     * Use the given list for the given role. Also add the role to the map of
     * listRoles.
     *
     * @param role
     *            the role which owns the list.
     * @param listConst
     *            the list constant for the list.
     * @param tableModel
     *            the list to add.
     */
    public void addRoleTableModel(RoleConstant role, 
            DialogObjectConstant listConst, 
            TableModel<? extends ListRow> tableModel) {
        setTableModel(role, tableModel);
        listRoles.put(listConst, role);
    }

    /**
     * Copy the given field value to all other fields of the same attribute.
     *
     * @param field
     *            the field with the input value.
     * @param value
     *            the field value to copy.
     */
    public void copyToEquivalentFields(DialogObjectConstant field, Object value) {
        AttributeConstant attribute = field.getAttribute();
        for (DialogObjectConstant otherField : getAttributeFields(attribute)) {
            if (field != otherField) {
                try {
                    Object otherValue = value;
                    if (ViewModelImpl.hasConverter(field)) {
                        Object attributeValue = convertToModelInternal(field, value);
                        otherValue = convertToViewInternal(otherField, attributeValue);
                    }
                    setField(otherField, otherValue);
                    setChanged(otherField, false); // Reset the changed status of the field.
                } catch (ConvertException e) {
                    // TODO what?
                }
            }
        }
    }

    @Override
    public Collection<DialogObjectConstant> getRoleFields(RoleConstant role) {
        Collection<DialogObjectConstant> roleFields = fieldsForRole.get(role);
        if (roleFields == null) {
            roleFields = new HashSet<DialogObjectConstant>();
            fieldsForRole.put(role, roleFields);
        }
        return roleFields;
    }

    @Override
    public void clearViewModel() {
        for (DialogObjectConstant field : getFields().keySet()) {
            setFieldValue(field, null);
        }
        for (TableModel<? extends ListRow> tableModel : tableModels.values()) {
            tableModel.clear();
        }
        for (TreeModel<? extends TreeNode, ? extends ListRow> treeModel : treeModels.values()) {
        	treeModel.clear();
        }
        changedFields.clear();
    }

    @Override
    public boolean isChanged(DialogObjectConstant field) {
        if (changedFields.contains(field)) {
            return true;
        }
        return false;
    }

    @Override
    public void setChanged(DialogObjectConstant field, boolean changed) {
        if (changed) {
            changedFields.add(field);
        } else {
            changedFields.remove(field);
        }
    }

    @Override
    public Collection<DialogObjectConstant> getChangedFields() {
        return Collections.unmodifiableCollection(changedFields);
    }

    @Override
    public Collection<AttributeConstant> getChangedAttributes(RoleConstant role) {
        Set<AttributeConstant> changedAttributes =
                new HashSet<AttributeConstant>();
        for (DialogObjectConstant field : changedFields) {
            if (field.getAttribute().getAttributeRole().equals(role)) {
                changedAttributes.add(field.getAttribute());
            }
        }
        return changedAttributes;
    }

    @Override
    public Object getFieldValue(DialogObjectConstant field) {
        Object fieldValue = getField(field);
        if (ViewModelImpl.hasConverter(field)) {
            fieldValue = convertToModel(field, fieldValue);
        }
        return fieldValue;
    }

    @Override
    public void setFieldValue(DialogObjectConstant field, Object fieldValue) {
        if (log.isTraceEnabled()) {
            log.trace("setFieldValue: " + field + "[" + fieldValue + "]");
        }
        if (ViewModelImpl.hasConverter(field)) {
            fieldValue = convertToView(field, fieldValue);
        }
        setField(field, fieldValue);
        setChanged(field, false); // Reset the changed status of the field.
    }

    /**
     * Set all field values for the given list row. The values are set in the
     * model for the edit fields.
     *
     * @param row
     *            - the row containing the field values to set.
     */
    public void setFieldValues(ListRow row) {
        for (DialogObjectConstant field : row.getFields()) {
            setField(field, row.getValue(field));
            setChanged(field, false); // Reset the changed status of the field.
            copyToEquivalentFields(field, row.getValue(field));
        }
    }

    /**
     * Clear all field values for the given list row.
     *
     * @param row
     *            - the row containing the field values to set.
     */
    public void clearFieldValues(ListRow row) {
        for (DialogObjectConstant field : row.getFields()) {
            setField(field, null);
            setChanged(field, false);
            copyToEquivalentFields(field, null);
        }
    }

    @Override
    public Collection<DialogObjectConstant> getAttributeFields(
            AttributeConstant attribute) {
        Collection<DialogObjectConstant> attributeFields =
                new HashSet<DialogObjectConstant>();
        Collection<DialogObjectConstant> roleFields =
                getRoleFields(attribute.getAttributeRole());
        for (DialogObjectConstant roleField : roleFields) {
            if (roleField.getAttribute() == attribute) {
                attributeFields.add(roleField);
            }
        }
        return attributeFields;
    }

    @Override
    public <T extends ListRow> List<T> getDisplayList(RoleConstant role) {
        TableModel<T> tableModel = getTableModel(role);
        if (tableModel == null) {
        	return null;
        }
        return tableModel.getTableView();
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ListRow> TableModel<T> getTableModel(RoleConstant role) {
        TableModel<T> tableModel = (TableModel<T>) tableModels.get(role);
        return tableModel;
    }

    @Override
    public <T extends ListRow> void setTableModel(RoleConstant role,
            TableModel<T> tableModel) {
        tableModels.put(role, tableModel);
    }

    @SuppressWarnings("unchecked")
	@Override
    public <T extends TreeNode, L extends ListRow> TreeModel<T,L> getTreeModel(DialogObjectConstant attribute) {
    	return (TreeModel<T, L>) treeModels.get(attribute);
    }

    @Override
    public void setTreeModel(DialogObjectConstant role, TreeModel<? extends TreeNode, ? extends ListRow> treeModel) {
        treeModels.put(role, treeModel);
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    @Override
    public <M,T> AttributeConverter<M,T> getConverter(DialogObjectConstant field) {
        AttributeConverter<?,?> converter = null;
        if (field.getAttribute() != null) {
            String converterId = field.getAttribute().getConverterId();
            if (converterId != null) {
                converter = G9Spring.<AttributeConverter>getBean(AttributeConverter.class, converterId);
            }
            // Special case, use Date in the view model if the attribute type is Joda or Java time dates.
            // If the attribute already has a converter, it is chained by the Joda or Java time converter.
            if (DialogObjectConstantHelper.isJodaType(field)) {
                if (log.isDebugEnabled()) {
                    log.debug("Creating a new Joda DateTime converter for " + field + ", chained with " + converterId);
                }
                converter = new JodaDateTimeConverter(converter, field.getAttribute().getAttributeType());
            }
            if (DialogObjectConstantHelper.isLocalDate(field)) {
                if (log.isDebugEnabled()) {
                    log.debug("Creating a new LocalDate converter for " + field + ", chained with " + converterId);
                }
                converter = new LocalDateConverter((AttributeConverter<LocalDate, Date>) converter);
            }
            if (DialogObjectConstantHelper.isLocalTime(field)) {
                if (log.isDebugEnabled()) {
                    log.debug("Creating a new LocalTime converter for " + field + ", chained with " + converterId);
                }
                converter = new LocalTimeConverter((AttributeConverter<LocalTime, Date>) converter);
            }
            if (DialogObjectConstantHelper.isLocalDateTime(field)) {
                if (log.isDebugEnabled()) {
                    log.debug("Creating a new LocalDateTime converter for " + field + ", chained with " + converterId);
                }
                converter = new LocalDateTimeConverter((AttributeConverter<LocalDateTime, Date>) converter);
            }
            // Special case, use String in the view model if the widget is AutoComplete and the view type isn't String
            // If the attribute already has a converter, it is chained by the Number converter.
            if (String.class != field.getAttribute().getAttributeType() && DialogObjectConstantHelper.isAutoCompleteEntry(field)) {
                if (log.isDebugEnabled()) {
                    log.debug("Creating a new Number converter for " + field + ", chained with " + converterId);
                }
                converter = new NumberConverter(converter, field.getAttribute().getAttributeType());
            }
        }
        return (AttributeConverter<M, T>) converter;
    }

    @Override
    public FieldValidator getValidator(DialogObjectConstant field) {
        String validatorId = field.getAttribute().getValidatorId();
        return G9Spring.getBean(FieldValidator.class, validatorId);
    }

    private Object convertToModel(DialogObjectConstant field, Object fieldValue) {
        try {
            return convertToModelInternal(field, fieldValue);
        } catch (ConvertException e) {
            JVineController.getInstance(getDialogController())
                    .addConverterException(e);
            return null;
        }
    }

    /**
     * Apply the converter to the given field value, converting from view
     * representation to model representation.
     * @param <M> model
     * @param <T> target
     *
     * @param field
     *            - the field to convert.
     * @param fieldValue
     *            - the value to convert.
     * @return the converted value.
     * @throws ConvertException
     *             if conversion fails.
     */
    <M,T> M convertToModelInternal(DialogObjectConstant field, T fieldValue)
            throws ConvertException {
        AttributeConverter<M,T> converter = getConverter(field);
        FieldConvertContext ctx = getConvertContext(field);
        return converter.toModel(fieldValue, ctx);
    }

    private <M,T> T convertToView(DialogObjectConstant field, M fieldValue) {
        try {
            return convertToViewInternal(field, fieldValue);
        } catch (ConvertException e) {
            JVineController.getInstance(getDialogController())
                    .addConverterException(e);
            return null;
        }
    }

    /**
     * Apply the converter to the given field value, converting from model
     * representation to view representation.
     * @param <M> model
     * @param <T> target
     *
     * @param field
     *            - the field to convert.
     * @param fieldValue
     *            - the value to convert.
     * @return the converted value.
     * @throws ConvertException
     *             if conversion fails.
     */
    <M,T> T convertToViewInternal(DialogObjectConstant field, M fieldValue)
            throws ConvertException {
        AttributeConverter<M,T> converter = getConverter(field);
        FieldConvertContext ctx = getConvertContext(field);
        return converter.fromModel(fieldValue, ctx);
    }

    /**
     * Validate the given field value.
     *
     * @param field
     *            - the field to validate.
     * @param value
     *            - the field value to validate.
     * @return the result of the validation.
     */
    public Map<ValidationResult, ValidateContext> validateField(DialogObjectConstant field,
            Object value) {
        return validateField(null, field, value);
    }

    /**
     * Validate the given field value.
     *
     * @param listRow
     *            - the ListRow for the field.
     * @param field
     *            - the field to validate.
     * @param value
     *            - the field value to validate.
     * @return the result of the validation.
     */
    public Map<ValidationResult, ValidateContext> validateField(ListRow listRow,
            DialogObjectConstant field, Object value) {
        if (validationManager == null) {
            validationManager =
                    ValidationManagerFactory
                            .create(null, getDialogController());
        }
        ValidateContext ctx = getValidateContext(listRow, field);
        return validationManager.validate(field, value, Policy.ON_CHANGE, ctx);
    }

    /**
     * Check if the given field has a converter.
     *
     * @param field
     *            - the field in question.
     * @return true if the field has a converter.
     */
    static boolean hasConverter(DialogObjectConstant field) {
        if (field.getAttribute() != null) {
            if (field.getAttribute().getConverterId() != null) {
                return true;
            }
            // Special case, use Date in the view model if the attribute type is Joda DateTime
            if (DialogObjectConstantHelper.isJodaType(field)) {
                return true;
            }
            if (DialogObjectConstantHelper.isJavaTimeType(field)) {
                return true;
            }
            // Special case, use String in the view model if the widget is AutoComplete and the view type isn't String
            if (String.class != field.getAttribute().getAttributeType() && DialogObjectConstantHelper.isAutoCompleteEntry(field)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get the convert context for the given field. The convert contexts are
     * cached in the view model, and created as needed.
     *
     * @param field
     *            - the field in question.
     * @return the found convert context, or a new if none found.
     */
    FieldConvertContext getConvertContext(DialogObjectConstant field) {
        FieldConvertContext ctx = convertContexts.get(field);
        if (ctx == null) {
            ctx = new FieldConvertContext(getDialogController(), field);
            convertContexts.put(field, ctx);
        }
        return ctx;
    }

    /**
     * Check if the given field has an validator.
     *
     * @param field
     *            - the field in question.
     * @return true if the field has an validator.
     */
    public boolean hasValidator(DialogObjectConstant field) {
        if (field.getAttribute().getValidatorId() != null
                || applicationView.getDialogView(instance).isMandatory(field)) {
            return true;
        }
        return false;
    }

    /**
     * Get the validate context for the given field. The validate contexts are
     * cached in the view model, and created as needed.
     *
     * @param listRow
     *            the row in question
     * @param field
     *            - the field in question.
     * @return the found validate context, or a new if none found.
     */
    ValidateContext getValidateContext(ListRow listRow,
            DialogObjectConstant field) {
        ValidateContext ctx = validateContexts.get(field);
        if (ctx == null) {
            ctx = new ValidateContext(getDialogController(), listRow, field);
            validateContexts.put(field, ctx);
        }
        return ctx;
    }

    @Override
    public DialogController getDialogController() {
        return applicationView.getDialogController(instance);
    }

    @SuppressWarnings("javadoc")
	public <T extends DialogView> T getDialogView() {
    	return applicationView.getDialogView(instance);
    }
    
    @Override
    public FieldData getCurrentFieldData(RoleConstant role) {
        FieldData fieldData = new FieldData(role);
        Collection<DialogObjectConstant> roleFields = getRoleFields(role);
        for (DialogObjectConstant field : roleFields) {
            fieldData.setFieldValue(field.getAttribute(), getFieldValue(field));
        }
        return fieldData;
    }

    @Override
    public Collection<FieldData> getAllFieldData(RoleConstant role) {
        Collection<FieldData> fieldDataList = new ArrayList<FieldData>();
        TableModel<ListRow> tableModel = getTableModel(role);
        for (ListRow row : tableModel.getTableData()) {
            fieldDataList.add(getRowFieldData(row, role));
        }
        return fieldDataList;
    }

    @Override
    public FieldData getRowFieldData(ListRow row, RoleConstant role) {
        FieldData fieldData = new FieldData(role);
        for (DialogObjectConstant field : row.getFields()) {
            Object value = row.getValue(field);
            if (ViewModelImpl.hasConverter(field)) {
                value = convertToModel(field, value);
            }
            fieldData.setFieldValue(field.getAttribute(), value);
        }
        return fieldData;
    }

    /**
     * @return the PropertyManager for this dialog view model.
     */
    public DefaultPropertyManager getPropertyManager() {
        return propertyManager;
    }

    /**
     * Reset the list row selection to the previous state.
     *
     * @param role
     *            - the object selection role for the list
     */
    public void resetListRowSelection(RoleConstant role) {
        List<? extends ListRow> list = getTableModel(role).getTableData();
        if (listSelections.get(role) != null) {
            int rowNo = listSelections.get(role).intValue();
            if (log.isTraceEnabled()) {
                log.trace("Resetting to row: " + rowNo + ", selected rows: "
                        + getSelectedRowNos(role));
            }
            for (ListRow row : list) {
                row.setSelected(Boolean.FALSE);
            }
            ListRow row = list.get(rowNo);
            if (row != null) {
                row.setSelected(Boolean.TRUE);
            }
            // Clear edit fields
            int singleRow = getSingleSelectedRowNo(role);
            if (singleRow == -1) {
                listSelections.remove(role);
                clearFieldValues(row);
                JVineController.getInstance(getDialogController()).clearCurrent(
                        role);
            }
        } else {
            for (ListRow row : list) {
                row.setSelected(Boolean.FALSE);
            }
        }
    }

    /**
     * Set the new selected state for the given list row. Also updates the edit
     * fields and the current reference in the dialog controller.
     *
     * @param selected
     *            - true if the list row is to be selected
     * @param role
     *            - the object selection role for the list
     * @param rowNo
     *            - the row in question
     * @param emptyRow
     *            - an empty list row containing a collection of all the fields
     *            in a list row
     *
     * @return true if the selection was set, false if interrupted (NO_SELECT)
     */
    public boolean setListRowSelection(boolean selected, RoleConstant role,
            int rowNo, ListRow emptyRow) {
        List<? extends ListRow> list = getDisplayList(role);
        TableModel<ListRow> tableModel = getTableModel(role);
        if (selected && tableModel.getSelectionModel() == TableModel.SelectionModel.NO_SELECT) {
            log.trace("Attempting to select row, but current selection model of " + role + " prohibits selection.");
            tableModel.setSelected(list.get(rowNo), false);
            return false;
        }
        tableModel.setSelected(list.get(rowNo), selected);

        log.trace("Selection state: " + selected + ", row: " + rowNo);
        int singleSelectedRowNo = getSingleSelectedRowNo(role);
        log.trace("Single: " + singleSelectedRowNo);

        if (singleSelectedRowNo != -1) {
            listSelections.put(role, Integer.valueOf(singleSelectedRowNo));
        } else {
            listSelections.remove(role);
        }
        return true;
    }

    private int getSingleSelectedRowNo(RoleConstant role) {
        int numSelected = 0;
        int rowNo = 0;
        for (ListRow row : getTableModel(role).getTableView()) {
            if (row.isRowSelected()) {
                numSelected++;
                rowNo = getTableModel(role).getTableView().indexOf(row);
            }
        }
        return numSelected == 1 ? rowNo : -1;
    }

    private List<Integer> getSelectedRowNos(RoleConstant role) {
        List<Integer> selectedRows = new ArrayList<Integer>();
        for (ListRow row : getTableModel(role).getTableView()) {
            if (row.isRowSelected()) {
                selectedRows.add(getTableModel(role).getTableView().indexOf(row));
            }
        }
        return selectedRows;
    }

    @SuppressWarnings("javadoc")
	public void updateTableFieldValues(RoleConstant role) {
        TableModel<ListRow> tableModel = getTableModel(role);
        if (tableModel != null) {
        	if (tableModel.getSelectionCount() == 1) {
        		ListRow selectedRow = tableModel.getSelected().get(0);
        		setFieldValues(selectedRow);
        		JVineController.getInstance(getDialogController()).setCurrent(getRowFieldData(selectedRow, role));
        	} else {
        		if (!tableModel.getTableData().isEmpty()) {
        			ListRow firstRow = tableModel.getTableData().get(0);
        			clearFieldValues(firstRow);
        			JVineController.getInstance(getDialogController()).clearCurrent(role);
        		}
        	}
        }
    }

    /**
     * Get the DialogObjectConstant which has the given internal name.
     *
     * @param internalName the name to search for
     * @return the found constant, or null if not found
     */
    public DialogObjectConstant getFieldDialogObjectConstant(String internalName) {
        DialogObjectConstant retVal = null;
        for (DialogObjectConstant field : fields.keySet()) {
            if (field.getInternalName().equals(internalName)) {
                retVal = field;
            }
        }
        return retVal;
    }

}
