/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;

import no.esito.jvine.action.HookMethod;
import no.esito.jvine.controller.JVineController;
import no.esito.jvine.controller.OSNode;
import no.esito.jvine.model.CurrentRoleObject;
import no.esito.jvine.rpc.ActualParameter;
import no.esito.jvine.rpc.ActualParameter.ParameterType;
import no.esito.log.Logger;
import no.g9.client.core.controller.DialogController;
import no.g9.os.AttributeConstant;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.action.ActionTarget;

/**
 * @param <V> The action result
 */
public class DisplayableAction<V> extends ObtainableAction<V> {

    /** The logger */
    private static Logger log = Logger.getLogger(DisplayableAction.class);

    /**
     * Constructs a new displayable g9 action.
     *
     * @param actionType the action type
     * @param actionTarget the action target
     * @param actionTask the callable implementing the action task
     * @param resultClass the target class *
     * @param dialogController the dialog controller that initiated this action
     */
    protected DisplayableAction(ActionType actionType, ActionTarget actionTarget,
            ActionTask<V> actionTask, Class<V> resultClass,
            DialogController dialogController) {
        super(actionType, actionTarget, actionTask, resultClass,
                dialogController);
    }

    /** The displayable hooks (if any) */
    private List<Displayable> displayableHooks = new ArrayList<Displayable>();

    @Override
    protected void setActionHookList(ActionHookList<V> actionHookList) {
        super.setActionHookList(actionHookList);
        if (actionHookList != null) {
            displayableHooks = actionHookList.getDisplayableHooks();
        }
    }

    /**
     * Returns the list of the displayable hooks.
     *
     * @return the list of the displayable hooks, possibly an empty list 
     */
    @Override
    List<Displayable> getDisplayableHooks() {
        return displayableHooks;
    }

    /**
     * Class used to invoke the display action with the result of performing
     * this action.
     */
    private class Display implements Callable<Void> {
        /** The object to display */
        Object toDisplay;

        /**
         * Constructs a new display object.
         *
         * @param toDisplay the object to display
         */
        Display(Object toDisplay) {
            this.toDisplay = toDisplay;
        }

        @Override
        public Void call() throws Exception {
            if (!CANCELLED) {
                JVineController jVineController = JVineController.getInstance(getController());
                ActionTask<Void> displayActionTask = jVineController.getDisplayDialogTask(getActionTarget(), getActionType());
                displayActionTask.setTaskObject(toDisplay);
                displayActionTask.call();

                if (isSaveAction()) {
                    restoreCurrentInstances();
                }

                displayed();
                log.info(DisplayableAction.this + " displayed");
            }

            return null;
        }

    }

    private class Clear implements Callable<Void> {

        @Override
        public Void call() throws Exception {
            if (!CANCELLED) {
                ActionType actionType = getActionType();

                switch (actionType) {
				case FIND : // Fall through
				case FINDALL :
                    getController().clearKeepKeys(
                            (RoleConstant) getActionTarget());
					break;
				case INVOKE :
					clearRemoteTarget((RemoteServiceTarget) getActionTarget());
					break;
				case DELETE :
				    getController().clear((RoleConstant) getActionTarget(), true);
				    break;
				default:
					break;
				}
            }
            cleared();
            return null;
        }

    }

    private void clearRemoteTarget(RemoteServiceTarget target) {
        switch (getReturnType(target)) {
        case ATTRIBUTE:
            AttributeConstant attribute = getInvokeReturnAttribute(target);
            getController().setFieldValue(attribute, null);
            break;
        case ROLE:
            RoleConstant role = getInvokeReturnRole(target);
            getController().clear(role, true);
            break;
        default:
            break;
        }

    }

    private ParameterType getReturnType(RemoteServiceTarget target) {
        ParameterBinding<Object> returnParameter = target.getReturnParameter();
        Parameter<Object> actualParameter = returnParameter.getActualParameter();
        if (actualParameter instanceof ActualParameter<?>) {
            ActualParameter<?> tmp = (ActualParameter<?>) actualParameter;
            return tmp.getParameterType();
        }
        return null;
    }


    private RoleConstant getInvokeReturnRole(RemoteServiceTarget target) {
        ParameterBinding<Object> returnParameter = target.getReturnParameter();
        Parameter<Object> actualParameter = returnParameter.getActualParameter();
        if (actualParameter instanceof ActualParameter<?>) {
            ActualParameter<?> tmp = (ActualParameter<?>) actualParameter;
            return tmp.getRole();
        }
        return null;
    }

    private AttributeConstant getInvokeReturnAttribute(RemoteServiceTarget target) {
        ParameterBinding<Object> returnParameter = target.getReturnParameter();
        Parameter<Object> actualParameter = returnParameter.getActualParameter();
        if (actualParameter instanceof ActualParameter<?>) {
            ActualParameter<?> tmp = (ActualParameter<?>) actualParameter;
            return tmp.getAttribute();
        }
        return null;
    }

    @Override
    public V call() throws Exception {
        log.info("Starting execution of " + this);
        V result = null;
        try {
            checking.call();
            boolean keepCurrentInfo = isSaveAction();
            if (keepCurrentInfo) {
                saveCurrentInstances();
            }
            Object obtainResult = obtaining.call();

            getActionTask().setTaskObject(obtainResult);

            Clear clearing = new Clear();
            clearing.call();

            Display displaying = new Display(getActionTask().getTaskObject());

            displaying.call();

        } catch (Exception e) {
            failed(e);
        } finally {
            cleanUpHook();
        }

        return result;

    }


    private void restoreCurrentInstances() {
        for (CurrentRoleObject cro : currentInstanceList) {
            JVineController controller = JVineController
                    .getInstance(getController());
            OSNode<?> node = controller.getOSNode(cro.getRole());
            Collection<?> allInstances = node.getAllInstances();
            for (Object object : allInstances) {
                if (object.equals(cro.getCurrent())) {
                    getController().setCurrentInstance(cro.getRole(), object);
                    break;
                }
            }
        }
    }

    private void saveCurrentInstances() {
        currentInstanceList = new ArrayList<CurrentRoleObject>();
        String target = getActionTarget().toString();
        RoleConstant roleConst = getController().getOSConst(target);

        OSNode<?> targetNode = JVineController.getInstance(getController())
                .getOSNode(roleConst);
        currentInstanceList = targetNode.getCurrentObjectList();
        return;
    }

    private List<CurrentRoleObject> currentInstanceList;

    
    /**
     * The displayed hook. Invoked before the succeeded hook.
     *
     * @throws Exception re-throws exception from hook method.
     */
    void displayed() throws Exception {
        if (shouldInvokeHook() && !getDisplayableHooks().isEmpty()) {
            String methodName = "displayed";
            ThreadType threadType = getThreadType(methodName, new Class[0]);
            HookMethod<Void> displayHook = new HookMethod<Void>(methodName) {
                @Override
                public Void call() throws Exception {
                    for (Displayable hook : getDisplayableHooks()) {
                        hook.displayed();
                    }
                    return null;
                }
            };
            hookInvoker.execute(getApplicationController(), threadType,
                    displayHook);
        }
    }

    /**
     * The cleared hook. Invoked before the display hook.
     *
     * @throws Exception re-throws exception from hook method.
     */
    void cleared() throws Exception {
        if (shouldInvokeHook() && !getDisplayableHooks().isEmpty()) {
            String methodName = "cleared";
            ThreadType threadType = getThreadType(methodName, new Class[0]);
            HookMethod<Void> clearHook = new HookMethod<Void>(methodName) {
                @Override
                public Void call() throws Exception {
                    for (Displayable hook : getDisplayableHooks()) {
                        hook.cleared();
                    }
                    return null;
                }

            };
            hookInvoker.execute(getApplicationController(), threadType,
                    clearHook);
        }
    }
}