/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.util.concurrent.Callable;

import org.springframework.util.StringUtils;

import no.esito.jvine.action.ActionHelper;
import no.esito.jvine.action.HookMethod;
import no.esito.jvine.communication.ActionMessageCompilation;
import no.esito.jvine.communication.SystemMessageUtils;
import no.esito.jvine.controller.DialogInteractionBroker;
import no.esito.jvine.controller.JVineController;
import no.g9.client.core.communication.G9ActionPayload;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.controller.CallBackProvider;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogController;
import no.g9.client.core.controller.DialogSetupValue;
import no.g9.client.core.controller.ExternalDialogConstant;
import no.g9.support.ActionType;

/**
 * The performing state, performs the action task, applies action parameter
 * hooks and invokes the performed hook and returns the result.
 * @param <V> type
 */
class Performing<V> extends ActionStage<V> implements Callable<V> {

	/**
	 * @param g9Action action
	 */
	Performing(G9Action<V> g9Action) {
		super(g9Action);
	}

    private void doDialogOpenHook() {
        DialogInteractionBroker broker = DialogInteractionBroker.getInstance();

        DialogSetupValue<?> provided = (DialogSetupValue<?>) ga.getActionSetupValue();
        DialogSetupValue<?> setup = provided;
        DialogConstant calleeConst = (DialogConstant) ga.getActionTarget();

        if (ga.isInvokeHooksOn()) {
            setup = broker.doSetup(provided, ga.getDialogController(), calleeConst);
        }

        if (calleeConst instanceof ExternalDialogConstant) {
            ActionMessageCompilation targetCompilation = ((ExternalDialogConstant) calleeConst).getReactor();
            if (setup != null) {
                SystemMessage callBackMessage = null;
                if (setup instanceof CallBackProvider) {
                    String callBackKey = ga.getApplicationController().addCallBack(ga.getDialogController(), calleeConst, ((CallBackProvider) setup).getCallBack());
                    String applicationId = StringUtils.uncapitalize(ga.getApplicationController().getApplicationName());
                    callBackMessage = SystemMessage.CALLBACK_TEMPLATE.receiver(applicationId).payload(callBackKey);
                }
                G9ActionPayload payload = targetCompilation.getG9ActionPayload().payload((callBackMessage!= null ? callBackMessage.code() + "//" : "") +
                        SystemMessageUtils.codeSetupValue(setup.getSetupValue()));
                targetCompilation.setG9ActionPayload(payload);
            }
            targetCompilation.forward();
        } else {
            DialogController callee = ga.getApplicationController().getDialogController((DialogConstant) ga.getActionTarget());
            callee.setSetupValue(setup);
            callee.setup(setup);
            JVineController.getInstance(callee).setCaller(ga.getDialogController());
        }
    }

    /**
     * doActionParameterHooks
     */
    void doActionParameterHooks() {
        if (ActionHelper.isParameterizedAction(ga.getActionType()) && !ga.getActionTask().FLAG) {
            // If more actions support parameterization, this should
            // refactored to a switch.
            if (ga.getActionType() == ActionType.OPEN) {
                doDialogOpenHook();
            }
        }
    }

    /**
     * The performed hook, invoked after initialized() unless the
     * action has failed or the action is cancelled. This hook is invoked on a
     * dedicated worker thread by default. Use the annotation {@link ThreadInfo}
     * to control which thread this method is invoked on.
     *
     * @param result The result of performing the action
     * @return Void - e.g. null
     * @exception Exception re-throws exception from hook method
     */
    Void performed(final V result) throws Exception {
        if (G9Action.getLog().isDebugEnabled()) {
            G9Action.getLog().debug(this + " performed. Result: " + result);
        }

        if (ga.shouldInvokeHook()) {
            final String methodName = "performed";
            ThreadType threadType = ga.getThreadType(methodName, ga.getResultClass());
            HookMethod<Void> invocation = new HookMethod<Void>(methodName) {

                @Override
                public Void call() {
                    for (ActionHook<V> hook : ga.getActionHookList().getHooks()) {
                        hook.performed(result);
                    }
                    return null;
                }
            };
            ga.getHookInvoker().execute(ga.getApplicationController(), threadType, invocation);
        }
        return null;
    }

	@Override
    public V call() throws Exception {
	    V result = null;
	    if (ga.getActionType() == ActionType.OPEN) {
	        ga.getActionTask().setTaskObject(new Boolean(ga.getFlag()));
	    }
	    result = ga.getActionTask().call();
	    doActionParameterHooks();
	    performed(result);
	    G9Action.getLog().info(ga + " performed");
	    return result;
    }
}